#include <qsplitter.h>
#include <qtabwidget.h>

#include "centralwidget.h"
#include "infoaddr.h"

CentralWidget::CentralWidget(BastardObject *bo, QWidget *parent, char *name = 0) : QWidget(parent, name)
{
    bastard = bo;
    genInfo = NULL;
    
    hexWidget = NULL;
    infoPanel = NULL;
    
    vertSplit = NULL;
    horSplit = NULL;
    
    connect(bastard, SIGNAL(readyNewTicket(unsigned int, void *)), this, SLOT(newTicket(unsigned int, void *)));
    
    infoRequestTicket = 0;
    hexWidgetHighlight = 0;
    
    init();
}


void CentralWidget::init(void)
{
    t_Function2Widget func2widget;
    
    /* Get general target info if needed */
    if (!genInfo)
    {
	connect(bastard, SIGNAL(readyTargetInfo()), this, SLOT(init()));
	genInfo = bastard->getTargetInfo();
    
	if (!genInfo) return;
	else disconnect(bastard, SIGNAL(readyTargetInfo()), this, SLOT(init()));
    }
    
    /* Create the splitter that will divide the screen in 2 */
    vertSplit = new QSplitter(this);
    
    /* ... and a horzontal splitter that will separate the left part of the screen */
    horSplit = new QSplitter(Qt::Vertical, vertSplit);
    
    /* Create the hex edit view to the top-left */
    hexWidget = new QHexEdit(horSplit);
    hexWidget->setSource(genInfo->image, genInfo->info.size, 0);
    
    /* Create the info widget at the bottom-left */
    infoPanel = new InfoPanel(bastard, horSplit);
    infoPanel->setFixedHeight(infoPanel->minimumSizeHint().height());
        
    /* Create the asm edit view to the right */
    asmTable = new QTabWidget(vertSplit);
    connect(asmTable, SIGNAL(currentChanged(QWidget*)), this, SLOT(asmTabChanged(QWidget*)));
    
    /* The first tab created is the function that contain the entry point of the target */
    func2widget.widget = NULL;
    func2widget.ticket = bastard->getFunctionRVA(genInfo->info.entry);
    
    asmWidgetList.push_back(func2widget);
    
    resizeEvent(NULL);
    vertSplit->show();
}


void CentralWidget::resizeEvent(QResizeEvent *)
{
    if (vertSplit) vertSplit->setGeometry(0,0,width(),height());
}


void CentralWidget::newTicket(unsigned int t, void *m)
{
    list<t_Function2Widget>::iterator it, itFind;
    
    BastardFunction *func;
    InfoAddr *infoAddr;
    
    /* Check if we got a msg for InfoAddr */
    if (t == infoRequestTicket)
    {
	infoAddr = (InfoAddr *)m;
	infoPanel->setInfo(infoAddr);
	
	/* Update the state of the hexWidget */
	hexWidget->blockPaintEvent(true);
	hexWidget->removeHighlight(hexWidgetHighlight);
	hexWidgetHighlight = hexWidget->addHighlight(infoAddr->getPA(), infoAddr->getAddrLen() + infoAddr->getPA() - 1, QColor(205, 234, 197) , Qt::AndROP);
	hexWidget->jumpTo(infoAddr->getPA());
	hexWidget->blockPaintEvent(false);
	
	infoRequestTicket = 0;
	delete infoAddr;
	return;
    }
	
    /* Sanity check */
    if (!asmWidgetList.size()) return;
    
    /* Figure out if a asm widget got a message and dispatch it */
    for (it = asmWidgetList.begin(); it != asmWidgetList.end(); it++)
    {	
	if (t == (*it).ticket)
	{
	    func = (BastardFunction *)m;
	    (*it).ticket = 0;
	    (*it).func = func;
	    
	    /* The user clicked on a non-jump/call instruction */
	    if (!func)
	    {
		asmWidgetList.erase(it);
		return;
	    }
	    
	    /* Figure out if we have the function already in memory */
	    if (asmWidgetList.size() > 1)
	    {
		for (itFind = asmWidgetList.begin(); itFind != asmWidgetList.end(); itFind++)
		{
		    if (*((*it).func) == (*itFind).func && (*it).func != (*itFind).func)
		    {
			/* We have found a duplicate function, get rid of the newly created one */
			(*itFind).func->setTargetRVA((*it).func->getTargetRVA());
			delete (*it).func;
			asmWidgetList.erase(it);
		    
			/* ... and zoom on the target line */
			(*itFind).widget->setHighlight((*itFind).func->getTargetRVA());
			(*itFind).widget->jumpTo((*itFind).func->getTargetRVA());
			asmTable->showPage((*itFind).widget);
		    
			return;		    
		    }
		}
	    }
	    
	    /* It's a new function that we have not displayed so far, create the widget and zoom on the line */
	    (*it).widget = new QAsmEdit(bastard, asmTable);
	    asmTable->addTab((*it).widget, func->getName());
	    asmTable->showPage((*it).widget);
	    
	    (*it).widget->setSourceFunction(func);
	    (*it).widget->setHighlight(func->getTargetRVA());
	    
	    connect((*it).widget, SIGNAL(clickOnRVA(unsigned int, ButtonState)), this, SLOT(asmClickOnRVA(unsigned int, ButtonState)));
	    
	    return;
	}
    }
}


CentralWidget::~CentralWidget()
{
    list<t_Function2Widget>::iterator it;
    
    if (asmWidgetList.size())
    {
	for (it = asmWidgetList.begin(); it != asmWidgetList.end(); it++)
	{
	    if ((*it).func) delete (*it).func;
	}
    }
}


void CentralWidget::asmClickOnRVA(unsigned int fromRVA, ButtonState bs)
{
    t_Function2Widget func2widget;
    
    if (bs & Qt::ControlButton)
    {
	func2widget.widget = NULL;
	func2widget.ticket = bastard->tryJumpFrom(fromRVA);
    
	asmWidgetList.push_back(func2widget);
    }
    else
    {
	infoRequestTicket = bastard->getInfoAddr(fromRVA);
    }
}


void CentralWidget::asmTabChanged(QWidget *w)
{
    asmClickOnRVA(((QAsmEdit *)w)->getHighlight(), NoButton);
}
