#ifndef __BASTARDOBJECT_H__
#define __BASTARDOBJECT_H__

#include <qobject.h>

#include "bastardthread.h"
#include "bastardfunction.h"

extern "C" {
#include <bastard.h>
}

enum t_BastardState { Init, Ready, Paused, Working };

struct t_DisasmTarget
{
    QString *name;
    QString *arch;
    QString *format;
    QString *comp;
    QString *lang;
    QString *os;
    QString *asmbl;
};

struct t_CodeDump
{
    unsigned int rva;
    char *label;
    char *instr;
    char *comment;
};

struct t_RVAString
{
    unsigned int rva;
    char *str;
};

/* WARNING: this function must be only created once in the program and must be passed between class
   in order to use the bastard */
class BastardObject: public QObject
{
    Q_OBJECT
    
    public:
    
    BastardObject(QObject *parent, char *name=0);
    ~BastardObject() { }
    
    void startThread(QString);
    void loadTarget(QString);
    
    void customEvent(QCustomEvent *ev);
        
    /* Requests infornmation, if the information is not valid, will return a NULL
       If the information is invalid, those function will call themselves the bastard thread
       for the requesed information. Then an appropriate signal will be emitted. 
      this kind of information is so general that we dont need the ticket system*/
    vector<struct section> *getSectionList(void);
    vector<struct address> *getAddressList(void);
    vector<struct code> *getCodeList(void);
    vector<struct function> *getFunctionList(void);
    
    struct DISASM_TGT *getTargetInfo(void);
    
    /* If you want to send a message to the bastard thread and this function does not return
       valiable information, use this function */
    void SendBastardMsg( t_BastardMsg msg, void *arg, unsigned int t );
    
    /* Useful functions to deal with SendBastardMsg */
    // This function convert from a QString to a char *
    // This function creates a new memory space that is needed to be deleted later by the caller
    char *CopyQStringToChar(QString *src);
    
    // Creates a new memory region and copy src into it
    // The calller must delete the allocated memory later on
    char *CopyString(const char *src);
    
    /* Other functions that might be useful */
    t_BastardState getBastardState(void);
    
    /* Those function will create a change in the bastard database */
    void changeCodeComment(unsigned int rva, char *newComment);
    
    /* New, Load, Save, etc... */
    bool saveNeeded(void) { return needSave; }
    void disasmNewTarget( t_DisasmTarget *t );
    void saveAs(QString);
    void closeTarget(void);
    
    
    /* Those function uses the ticket system. They will return an unique id number
       and a signal will be emmited  "readyNewTicket(id_number, void *data)" when the task is complete*/
    unsigned int getCodeSection(struct section *);
    unsigned int getCodeFunction(BastardFunction *);
    unsigned int getFunctionRVA(unsigned int rva);
    unsigned int tryJumpFrom(unsigned int rva);
    unsigned int getInfoAddr(unsigned int rva);
    
    
    signals:
    /* Those signals are emmited when this particular info is now available */
    void readySectionList(void);
    void readyAddressList(void);
    void readyCodeList(void);
    void readyFunctionList(void);
    
    void readyNewTicket(unsigned int, void *);
    
    void readyTargetInfo(void);
    
    /* Those signals tell the state of the basard thread */
    void readyDisasmDone(void);
    void readyQuit(void);
    void idleThread(void);
    void workingThread(void);
    void createdThread(void);
    
    /* Those signals are emitted when there are change in the database */
    void codeCommentChanged(unsigned int, char *);
    
    /* Signmals related to load/save */
    void targetClosed(void);
    
    private:
    int threadPid;
    BastardThread *bastardThread;
    t_BastardState bastardState;
    
    vector<struct section> *listSection;
    vector<struct address> *listAddress;
    vector<struct code> *listCode;
    vector<struct function> *listFunction;
    
    struct DISASM_TGT *targetInfo;
    
    list <struct t_BastardMsgList> bastardMsgList;
    
    unsigned int ticketId; /* ID of the next ticket to be created */
    
    bool firstMsgFromThread;
    
    bool needSave;
    
    /* Private functions */
    bool TryDispatchNextBastardMsg(void);
    unsigned int genTicket(void);
    
};

#endif
