#include "bastardobject.h"

BastardObject::BastardObject(QObject *parent, char *name = 0) : QObject(parent, name)
{
    /* Init most of the var to unknown state */
    listSection = NULL;
    listAddress = NULL;
    listCode = NULL;
    listFunction = NULL;
    
    targetInfo = NULL;
    
    /* TickedId == 0 is reserved to general information gathering */
    ticketId = 1;
    
    firstMsgFromThread = true;
    
    needSave = false;
}


void BastardObject::startThread(QString installPath)
{
    // Create the thread and init the semaphore
    bastardThread = new BastardThread(this);
    BastardThread::semaphore = new QSemaphore(1);
    (*BastardThread::semaphore)++;
    
    // Start the thread
    bastardThread->setInstallPath(installPath);
    bastardThread->start();
}


// This function convert from a QString to a char *
// This function creates a new memory space that is needed to be deleted later by the caller
char *BastardObject::CopyQStringToChar(QString *src)
{
    char *ret;
    
    ret = new char[strlen(src->latin1())+1];
    strcpy(ret,src->latin1());
    
    return ret;
}

// Creates a new memory region and copy src into it
// The calller must delete the allocated memory later on
char *BastardObject::CopyString(const char *src)
{
    char *ret;
    
    ret = new char[strlen(src) + 1];
    strcpy(ret, src);
    
    return ret;
}



void BastardObject::customEvent(QCustomEvent *ev)
{    
    switch (ev->type())
    {
    case 12345:	// Ready message from the bastard thread
	threadPid = *((pid_t *)ev->data());
	bastardState = Ready;
	delete (pid_t *)ev->data();
	
	if  (firstMsgFromThread)
	{
	    emit createdThread();
	    firstMsgFromThread = false;
	}

	// We will try to dispatch the next message
	if (!TryDispatchNextBastardMsg())
	    emit idleThread();
	
	break;
	
	
	// Message sent from the bastard in reply from a message sent earlier
    case 12346:
	if (!((t_ReplyMsg*)ev->data())->ticketId)
	{
	    /* Dispatch general ticket stuff */
	    switch (((t_ReplyMsg*)ev->data())->msg)
	    {
	    case GET_DUMP_SECTION:
		listSection = (vector<struct section> *)(((t_ReplyMsg*)ev->data())->arg);
		emit readySectionList();
		break;
	    
	    case GET_DUMP_ADDRESS:
		listAddress = (vector<struct address> *)(((t_ReplyMsg*)ev->data())->arg);
		emit readyAddressList();
		break; 
	    
	    case GET_DUMP_CODE:
		listCode = (vector<struct code> *)(((t_ReplyMsg*)ev->data())->arg);
		emit readyCodeList();
		break;
		
	    case GET_DUMP_FUNCTION:
		listFunction = (vector<struct function> *)(((t_ReplyMsg*)ev->data())->arg);
		emit readyFunctionList();
		break; 
	    
		
	    case BASTARD_SYS_QUIT:
		emit readyQuit();
		break;
		
	    case BASTARD_TARGET_LOAD_BDB:
	    case BASTARD_DISASM_TARGET:
		emit readyDisasmDone();
		break;
	    
	    case BASTARD_GET_TARGET_INFO:
		targetInfo = (struct DISASM_TGT *)(((t_ReplyMsg*)ev->data())->arg);
		emit readyTargetInfo();
		break;
	    
		default:
		break;
	    }
	}
	else
	{
	    /* Ticket message */
	    emit readyNewTicket(((t_ReplyMsg*)ev->data())->ticketId, (void *)(((t_ReplyMsg*)ev->data())->arg));
	}
	
	// Get rid of the reply struct
	delete (t_ReplyMsg *)ev->data();
	
	break;
	    	    
	
    default:
	break;
    }
}


bool BastardObject::TryDispatchNextBastardMsg(void)
{
    t_BastardMsgList bastardMsg;
    /* We can dispatch the nest message only if
     1) We have an actual message!
     2) The bastard thread is ready for the next message */
    if (bastardMsgList.size() && bastardState == Ready)
    {
	bastardState = Working;
	emit workingThread();
	
	bastardMsg = *bastardMsgList.begin();
	bastardMsgList.pop_front();
	
	// Copy the top message to the BastardThread class
	BastardThread::msg = bastardMsg.msg;
	BastardThread::msgArg = bastardMsg.arg;
	BastardThread::ticketId = bastardMsg.ticketId;
		
	// Tell the bastard thread that it can process the message
	(*BastardThread::semaphore)--;
	
	// Was able to dispatch a message
	return true;
    }
    else
	return false;
}

void BastardObject::SendBastardMsg( t_BastardMsg msg, void *arg, unsigned int t )
{
    t_BastardMsgList bastardMsg;
    bastardMsg.msg = msg;
    bastardMsg.arg = arg;
    bastardMsg.ticketId = t;
    
    // We will put the message at the end of the list...    
    bastardMsgList.push_back(bastardMsg);

    // We will *try* to dispatch the next message
    TryDispatchNextBastardMsg();
}


void BastardObject::loadTarget(QString t)
{
    SendBastardMsg(BASTARD_TARGET_LOAD_BDB, CopyQStringToChar(&t), 0);
    
    needSave = false;
}


t_BastardState BastardObject::getBastardState(void)
{
    return bastardState;
}


vector<struct section> *BastardObject::getSectionList(void)
{
    if (listSection) return listSection;
    
    SendBastardMsg(GET_DUMP_SECTION, NULL, 0);
    
    return NULL;
}


vector<struct address> *BastardObject::getAddressList(void)
{
    if (listAddress) return listAddress;
    
    SendBastardMsg(GET_DUMP_ADDRESS, NULL, 0);
    
    return NULL;
}


vector<struct code> *BastardObject::getCodeList(void)
{
    if (listCode) return listCode;
    
    SendBastardMsg(GET_DUMP_CODE, NULL, 0);
    
    return NULL;
}


unsigned int BastardObject::genTicket(void)
{
    ticketId ++;    
    return ticketId - 1;    
}



vector<struct function> *BastardObject::getFunctionList(void)
{
    if (listFunction) return listFunction;
    
    SendBastardMsg(GET_DUMP_FUNCTION, NULL, 0);
    
    return NULL;
}


struct DISASM_TGT *BastardObject::getTargetInfo(void)
{
    if (targetInfo) return targetInfo;
    
    SendBastardMsg(BASTARD_GET_TARGET_INFO, NULL, 0);
    
    return NULL;
}


unsigned int BastardObject::getCodeSection(struct section *s)
{
    struct section *sec = new section;
    memcpy(sec, s, sizeof(struct section));
    
    SendBastardMsg(GET_CODE_SECTION, sec, ticketId);
    
    return genTicket();
}


unsigned int BastardObject::getCodeFunction(BastardFunction *f)
{
    BastardFunction *func = new BastardFunction(f);
    
    SendBastardMsg(GET_CODE_FUNCTION, func, ticketId);
    
    return genTicket();
}


unsigned int BastardObject::getFunctionRVA(unsigned int rva)
{
    unsigned int *r = new unsigned int;
    *r = rva;
    
    SendBastardMsg(GET_FUNCTION_RVA, r, ticketId);
    
    return genTicket();
}


unsigned int BastardObject::tryJumpFrom(unsigned int rva)
{
    unsigned int *r = new unsigned int;
    *r = rva;
    
    SendBastardMsg(TRY_JUMP_FROM, r, ticketId);
    
    return genTicket();
}


unsigned int BastardObject::getInfoAddr(unsigned int rva)
{
    unsigned int *r = new unsigned int;
    *r = rva;
    
    SendBastardMsg(GET_INFO_ADDR, r, ticketId);
    
    return genTicket();
}


void BastardObject::changeCodeComment(unsigned int rva, char *newComment)
{
    t_RVAString *send = new t_RVAString;
    
    send->str = new char[strlen(newComment)+1];
    strcpy(send->str, newComment);
    send->rva = rva;
    
    SendBastardMsg(CHANGE_CODE_COMMENT, send, 0);
    
    needSave = true;
    
    emit codeCommentChanged(rva, newComment);
}


void BastardObject::disasmNewTarget( t_DisasmTarget *t )
{
    if (t)
    {
	/* Send all the necessary command to load the binary, set all the target information and
	   then do the actual dissasembly */
	SendBastardMsg(BASTARD_TARGET_LOAD, CopyQStringToChar(t->name), 0);
	SendBastardMsg(BASTARD_TARGET_SET_FORMAT, CopyQStringToChar(t->format), 0);
	SendBastardMsg(BASTARD_TARGET_SET_ARCH, CopyQStringToChar(t->arch), 0);
	SendBastardMsg(BASTARD_TARGET_SET_COMP, CopyQStringToChar(t->comp), 0);
	SendBastardMsg(BASTARD_TARGET_SET_LANG, CopyQStringToChar(t->lang), 0);
	SendBastardMsg(BASTARD_TARGET_SET_ASM, CopyQStringToChar(t->asmbl), 0);
	SendBastardMsg(BASTARD_TARGET_SET_OS, CopyQStringToChar(t->os), 0);
	SendBastardMsg(BASTARD_TARGET_APPLY_FORMAT, NULL, 0);	
	SendBastardMsg(BASTARD_DISASM_TARGET, CopyString("full"), 0);
	
	/* We disassembled the file, we have something to save */
	needSave = true;
	
	delete t;
    }
}


void BastardObject::saveAs(QString bdbFilename)
{
    SendBastardMsg(BASTARD_TARGET_SAVE_DB_AS, CopyQStringToChar(&bdbFilename), 0);
    
    needSave = false;
}


void BastardObject::closeTarget(void)
{
    if (listSection)
    {
	delete listSection;
	listSection = NULL;
    }
    
    if (listAddress)
    {
	delete listAddress;
	listAddress = NULL;
    }
    
    if (listCode)
    {
	delete listCode;
	listCode = NULL;
    }
    
    if (listFunction)
    {
	delete listFunction;
	listFunction = NULL;
    }
    
    targetInfo = NULL;
    
    emit targetClosed();
}
