/****************************************************************************
** ui.h extension file, included from the uic-generated form implementation.
**
** If you wish to add, delete or rename slots use Qt Designer which will
** update this file, preserving your code. Create an init() slot in place of
** a constructor, and a destroy() slot in place of a destructor.
*****************************************************************************/

#include <qsettings.h>
#include <qdragobject.h>
#include <qapplication.h>
#include <qfiledialog.h>
#include <qregexp.h>
#include <qmessagebox.h>

#include "bastardobject.h"
#include "loadtargetdialog.h"
#include "QHexEdit/qhexedit.h"
#include "QAsmEdit/qasmedit.h"
#include "centralwidget.h"

extern "C" {
#include <bastard.h>
}

QString appName = QString("qBastard Disassembly Environment");


/* Very useful function that will return a pixmap from the image collection */
QPixmap lookupPixmap ( const QString name )
{
    const QMimeSource* mime_source = QMimeSourceFactory::defaultFactory()->data( name );
    if ( mime_source == 0 ) return QPixmap();
    
    QPixmap pixmap;
    QImageDrag::decode( mime_source, pixmap );
    
    return pixmap;
}




void BastardGUI::init()
{
    QSettings set;
    bastard = NULL;
    
    // Set the status bar
    bastardStateTextLabel = new QLabel("", statusBar());
    updateStatusBar();
    statusBar()->addWidget(bastardStateTextLabel, 0, TRUE);
    
    // Create the bastard object (i.e. the class that is used to communicate with the bastard)
    bastard = new BastardObject(this, "BastardObject");

    connect(bastard, SIGNAL(workingThread()), this, SLOT(updateStatusBar()));
    connect(bastard, SIGNAL(idleThread()), this, SLOT(updateStatusBar()));
    connect(bastard, SIGNAL(createdThread()), this, SLOT(threadReady()));
    connect(bastard, SIGNAL(readyDisasmDone()), this, SLOT(displayCentralWidget()));
    connect(bastard, SIGNAL(readyQuit()), qApp, SLOT(quit()));
    connect(bastard, SIGNAL(targetClosed()), this, SLOT(targetClosed()));
    
    // The settings file does not exsists, we will create it with the "factory default" information
    if (!set.readEntry("/qbastard/Path/InstallPath"))
    {
	set.writeEntry("/qbastard/Path/InstallPath", "/usr/local/bastard");
	set.writeEntry("/qbastard/Path/DatabasePath", QString("%1/.bastard").arg(getenv("HOME")));
    }
    
    bastard->startThread(set.readEntry("/qbastard/Path/InstallPath"));
    

    // Preload some dialog
    loadTargetDialog = new LoadTargetDialog(this, "loadTargetDialog");
    connect(loadTargetDialog, SIGNAL(newTarget(t_DisasmTarget *)), this, SLOT(newTarget(t_DisasmTarget *)));
	    
    //disasmProgressDialog = new DisasmProgressDialog(this, "BastardGUI::disasmTargetDialog", FALSE);
    configDialog = new ConfigDialog(this, "configDialog", TRUE);
    
    // ??? init is never called when configDialog is created... WTF?
    configDialog->init();
    
    bdbFilename = QString::null;
    targetName = QString::null;
    
    databaseDlg = NULL;
    centralWidget = NULL;
    aboutDlg = NULL;
}


void BastardGUI::fileOpen()
{
    QSettings set;
    
    QString s = QFileDialog::getSaveFileName(
	    set.readEntry("/qbastard/Path/DatabasePath"),
	    tr("Bastard Database files (*.bdb)"),
	    this,
	    "loadCommonDlg",	// Inernal name
	    tr("Load database"));
    
    if (s != QString::null) loadTarget(s);
}



void BastardGUI::fileSave()
{
    QString msg;
    
    if (bdbFilename == QString::null)
    {
	fileSaveAs();
	return;
    }
    
    bastard->saveAs(bdbFilename);
    
    msg = bdbFilename + QString(tr(" saved."));
    statusBar()->message(msg, 3000);
}


void BastardGUI::fileSaveAs()
{
    QSettings set;
    
    /* Getting the default path and file name for the file */
    QString def;
    if (bdbFilename != QString::null)
    {
	def = bdbFilename;
    }
    else
    {
	/* We will get the filename (i.e. strip the path) and add a .bdb */
	def = set.readEntry("/qbastard/Path/DatabasePath");
	QRegExp rx("/([^/]+)$");
	rx.search(targetName);
	def = def + QString("/");
	def = def + rx.cap(1);
	def = def + QString(".bdb");
    }
	 
    bdbFilename = QFileDialog::getSaveFileName(
	    def,			
	    tr("Bastard Database files (*.bdb)"),
	    this,
	    "saveCommonDlg",	// Inernal name
	    tr("Save as..."));
    
    if (!(bdbFilename == QString::null)) fileSave();
    else bdbFilename = def;
    
    updateTitleBar();
}

void BastardGUI::filePrint()
{

}

void BastardGUI::fileExit()
{
    if (bastard->saveNeeded())
    {
	switch (QMessageBox::information(this,
					 QString(tr("Unsaved changes!")),
					 QString(tr("Your current target contains unsaved changes. Do you want to save before exiting qBastard?")),
					 QString(tr("&Yes")),
					 QString(tr("&No")),
					 QString(tr("&Cancel")),
					 0, 2))
	{
	case 0:
	    fileSave();
	    break;
	    
	case 1:
	    break;
	    
	case 2:
	    return;
	    
	default:
	    return;
	}
    }
    
    // Send the quit message to the bastard thread
    bastard->SendBastardMsg(BASTARD_SYS_QUIT, NULL, 0);
}

void BastardGUI::helpIndex()
{

}

void BastardGUI::helpContents()
{

}

void BastardGUI::helpAbout()
{
    if (!aboutDlg)
	aboutDlg = new AboutDialog(this);
    
    aboutDlg->show();
}


void BastardGUI::displayCentralWidget(void)
{
    /* Create and display the central widget */
    centralWidget = new CentralWidget(bastard, this);
    setCentralWidget(centralWidget);
    centralWidget->show();
        
    /* Disassembly is done, we will activate save features and stuff */
    viewDatabaseAction->setEnabled(TRUE);
    fileSaveAction->setEnabled(TRUE);
    fileSaveAsAction->setEnabled(TRUE);
    fileCloseAction->setEnabled(TRUE);
    
    /* Restore the normal mouse cursor */
    QApplication::restoreOverrideCursor();
    
    /* Set the target name in the application title bar */
    updateTitleBar();
}



void BastardGUI::showConfigDialog()
{
    configDialog->exec();
}


void BastardGUI::updateStatusBar()
{
    if (bastard)
    {
	switch (bastard->getBastardState())
	{
	case Init:
	    bastardStateTextLabel->setText(tr("Bastard: Init"));
	    break;
	    
	case Working:
	    bastardStateTextLabel->setText(tr("Bastard: Working"));
	    break;
	
	case Ready:
	    bastardStateTextLabel->setText(tr("Bastard: Ready"));
	    break;
	
	case Paused:
	    bastardStateTextLabel->setText(tr("Bastard: Paused"));
	    break;
	}
    }
    else
	bastardStateTextLabel->setText(tr("Bastard: Init"));
}


void BastardGUI::showDatabaseDump()
{
    if (!databaseDlg)
	databaseDlg = new DatabaseDumpDialog(bastard, this, "databaseDlg");
    
    databaseDlg->show();
}


void BastardGUI::loadTarget( QString t )
{
    if (fileClose())
    {
	bastard->loadTarget(t);
	targetName = QString::null;
	bdbFilename = t;
    
	/* Set the mouse icon to busy state */
	QApplication::setOverrideCursor(Qt::WaitCursor);
    }
}



void BastardGUI::threadReady( void )
{
    /* Enable the File...New and File...Open operations */
    newTargetAction->setEnabled(TRUE);
    fileOpenAction->setEnabled(TRUE);
}


void BastardGUI::newTarget( t_DisasmTarget *t )
{
    if (t)
    {
	targetName = *(t->name);
	
	bastard->disasmNewTarget(t);

	/* Set the mouse icon to busy state */
	QApplication::setOverrideCursor(Qt::WaitCursor);

	updateTitleBar();
    }
}


void BastardGUI::fileNew()
{
    loadTargetDialog->exec();
}


void BastardGUI::closeEvent( QCloseEvent * )
{
    fileExit();
}


void BastardGUI::updateTitleBar()
{
    QString title = appName;
    
    if (bdbFilename == QString::null && targetName != QString::null)
	title = title + QString(" - [New Target]");
    
    if (bdbFilename) 
	title = title + QString(" - ") + bdbFilename;
    
    setCaption(title);    
}


void BastardGUI::targetClosed()
{
    if (databaseDlg)
    {
	delete databaseDlg;
	databaseDlg = NULL;
    }
    
    if (centralWidget)
    {
	delete centralWidget;
	centralWidget = NULL;
    }
    
    bdbFilename = QString::null;
    targetName = QString::null;
    
    updateTitleBar();
}


bool BastardGUI::fileClose()
{
    /* Check to see if we have already a target in memory, if we find one and it needs to be saved,
       tell the user to do so before loading a new target */
    if ((bdbFilename != QString::null || targetName != QString::null) && bastard->saveNeeded())
    {
	switch (QMessageBox::information(this,
					 QString(tr("Unsaved changes!")),
					 QString(tr("Your current target contains unsaved changes. Do you want to save now?")),
					 QString(tr("&Yes")),
					 QString(tr("&No")),
					 QString(tr("&Cancel")),
					 0, 2))
	{
	case 0:
	    fileSave();
	    break;
	    
	case 1:
	    break;
	    
	case 2:
	    return false;
	    
	default:
	    return false;
	}	
    }
    
    viewDatabaseAction->setEnabled(FALSE);
    fileSaveAction->setEnabled(FALSE);
    fileSaveAsAction->setEnabled(FALSE);
    fileCloseAction->setEnabled(FALSE);
    
    bastard->closeTarget();

    targetName = QString::null;
    bdbFilename = QString::null;
    
    updateTitleBar();
    
    return true;
}
