#include <stdlib.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qimage.h>

#include "qhexeditwidget.h"

QHexEditWidget::QHexEditWidget(QWidget *parent, const char *name = 0) : QWidget(parent, name)
{
    init();
    
    vScroll = NULL;
    hScroll = NULL;
    
}


QHexEditWidget::QHexEditWidget(QScrollBar *v, QScrollBar *h, QWidget *parent, const char *name = 0) : QWidget(parent, name)
{
        
    vScroll = v;
    hScroll = h;

    connect(vScroll, SIGNAL(valueChanged(int)), this, SLOT(scrollChanged(int)));
    connect(hScroll, SIGNAL(valueChanged(int)), this, SLOT(scrollChanged(int)));
    
    init();
}


void QHexEditWidget::init(void)
{
    needRedraw = false;
    blockRedraw = false;
    
    source = NULL;
    
    x = 0;
    y = 0;
    nbLine = 0;
        
    font = QFont("Courier", 10);
    QFontMetrics fontInfo = QFontMetrics(font);
    fontWidth = fontInfo.maxWidth();
    fontHeight = fontInfo.lineSpacing();
    
    nextHiNo = 1;
    
    calcDisplayNeed();
    adjustScrollBar();
    updateFontCache();
    
    setBackgroundMode(PaletteBase);
}


void QHexEditWidget::paintEvent(QPaintEvent *)
{
    unsigned int lenAt = 0, lenAtMax;
    unsigned int addrAt = baseAddr;
    int i, j, lineMin, lineMax, posx;
    unsigned char buf[128];
    
    list<t_HighlightBlock>::iterator it;
    unsigned int hiStart, hiEnd, hiLen, hiLenX;
    
    /* Check if redraw are blocked */
    if (blockRedraw)
    {
	needRedraw = true;
	return;
    }
    
    /* TO DO: We should keep the whole QPixamp generated from frame to frame,
       while scrolling we can only have to update 10% of the QPixmap make some big optimization */
        
    /* We will draw in a pixmap and later we will blit on the widget to remove flickering */
    QPixmap pm(size());
    pm.fill(); // Set it to a white image
    
    QPainter paint;
    paint.begin(&pm, this);

    /* setSource() has not beed called so far, we have nothing to draw */
    if (!source) return;
    
    /* Determine which lines that will actually being printed on the widget */
    lineMin = y / fontHeight;
    lineMax = (y + height()) /  fontHeight + 2;  /* This value is +1, faster for the for loops later */
    if (lineMax > nbLine) lineMax = nbLine;
    lenAtMax = lineMax * 16 - 1;

    /* Display all the addresses */
    addrAt = lineMin * 16 + baseAddr;
    
    /* Get the first addres in hex */
    for (i = 7; ; i--)
    {
	buf[i] = addrAt & 0xF;
	
	if (buf[i] <= 9) buf[i] += 48;
	else buf[i] += 55;
	
	addrAt = addrAt >> 4;
	
	if (i == 0) break;
    }
    
    /* Write all the addresses */
    for (j = 0; j < (lineMax - lineMin) * fontHeight; j += fontHeight)
    {
	for (i = 0; i < 8; i++) bitBlt(&pm, i * fontWidth - x, j, &fontCacheBlack[buf[i]]);
	
	/* Calc the next addr based on the current hex output */
	for (i = 6; ; i--)
	{
	    buf[i]++;
	    if (buf[i] == 58) buf[i] = 65;	// 9 + 1 = A
	    if (buf[i] == 71) buf[i] = 48;	// F + 1 = 10, so we will make another turn in this for loop
	    else break;
	}		
    }
    
    /* Display the line between the addresses and the hex dump */
    paint.drawLine(fontWidth*10 - x, 0, fontWidth*10 - x, height());
    
    /* Display in hexadecimal */
    lenAt = lineMin * 16;
    for (j = 0; j < (lineMax - lineMin) * fontHeight; j += fontHeight)
    {
	posx = 12 * fontWidth - x;
	
	for (i = 0; i < 16; i++)
	{
	    /* Convert the data into hex */
	    buf[0] = source[lenAt] >> 4;
	    if (buf[0] <= 9) buf[0] += 48;
	    else buf[0] += 55;
	    
	    buf[1] = source[lenAt] & 0xF;
	    if (buf[1] <= 9) buf[1] += 48;
	    else buf[1] += 55;
	    
	    /* Figure out which color we will display this column and display it*/
	    if (i % 2) bitBlt(&pm, posx, j, &fontCacheBlue[buf[0]]);
	    else bitBlt(&pm, posx, j, &fontCacheBlack[buf[0]]);
	    
	    posx += fontWidth;
	    
	    if (i % 2) bitBlt(&pm, posx, j, &fontCacheBlue[buf[1]]);
	    else bitBlt(&pm, posx, j, &fontCacheBlack[buf[1]]);

	    posx += fontWidth;
	    
	    /* Add an additionnal space at the middle of the hex dump */
	    if (i == 7)  posx += (2*fontWidth);
	    else posx += fontWidth;
	    
	    lenAt++;
	    if (lenAt > sourceLen) break;
	}
    }
    
    
    /* Display the line between the hex dump and the ascii dump */
    paint.drawLine(fontWidth*62 - x, 0, fontWidth*62 - x, height());
    
    /* Display the ascii dump */
    lenAt = lineMin * 16;    
    for (j = 0; j < (lineMax - lineMin) * fontHeight; j += fontHeight)
    {
	posx = 64 * fontWidth - x;
	
	for (i = 0; i < 16; i++)
	{
	    /* Convert the character to '.' if it is a control character (ASCII < 32) */
	    if (source[lenAt] >= 32)	 buf[0] = source[lenAt];
	    else buf[0] = '.';
	    
	    bitBlt(&pm, posx, j, &fontCacheBlack[buf[0]]);
	    posx += fontWidth;
	    
	    lenAt++;
	    if (lenAt > sourceLen) break;
	}
	
	if (lenAt > sourceLen) break;
    }
    
    if (hiList.size())
    {
	lenAt = lineMin * 16;
	
	for (it = hiList.begin(); it != hiList.end() && (*it).start <= lenAtMax; it++)
	{
	    if ((*it).start <= lenAtMax && (*it).end >= lenAt)
	    {
		/* We have found an hightlight that is on the screen, display it */
		if ((*it).start < lenAt) hiStart = lenAt;
		else hiStart = (*it).start;
		
		if ((*it).end > lenAtMax) hiEnd = lenAtMax;
		else hiEnd = (*it).end;
		
		paint.setRasterOp((*it).rop);
		paint.setBrush(QBrush((*it).color));
		paint.setPen(Qt::NoPen);
		
		while(hiEnd >= hiStart)
		{
		    /* TO DO: When there is very long highlight, we can reduce the number of call to drawRect... */
		    /* Warning: the following lines of code can give you a headache :) */
		    /* Highlight in the hex dump */
		    
		    /* hiStart and hiEnd are on the same line */
		    if ((hiStart/16) == (hiEnd/16)) hiLen = hiEnd - hiStart;
		    else hiLen = 15 - (hiStart % 16);
		    
		    /* Get hiLen in term of device x */
		    hiLenX = hiLen * fontWidth * 3 + 2 * fontWidth - x;
		    
		    /* Add a character if the highlight pass the middle of the hex dump */
		    if (((hiStart/8)%2) != (((hiStart+hiLen)/8)%2)) hiLenX += fontWidth;
		    
		    paint.drawRect((hiStart % 16) * 3 * fontWidth + fontWidth * 12 - x + (((hiStart/8)%2) ? fontWidth : 0),	
				   ((hiStart - lenAt)/16)*fontHeight,
				   hiLenX,
				   fontHeight);
		    
		    /* Highlight in the ASCII dump */
		    paint.drawRect((hiStart % 16) * fontWidth + fontWidth * 64 - x,
				   ((hiStart - lenAt)/16)*fontHeight,
				   (((hiStart % 16) + (hiEnd - hiStart)) <= 15) ? (((hiEnd - hiStart) % 16) * fontWidth + fontWidth) : ((16 * fontWidth) - (hiStart % 16) * fontWidth),
				   fontHeight);
		    
		    
		    /* hiStart will now point at the beginning of the next line */
		    hiStart += 16;
		    hiStart -= (hiStart % 16);
		}
		
	    }
	}
    }

    paint.end();
    bitBlt(this, 0, 0, &pm);
}


void QHexEditWidget::resizeEvent(QResizeEvent *)
{
    adjustScrollBar();
}


void QHexEditWidget::setSource(void *s, unsigned int len, unsigned int bAddr)
{
    source = (unsigned char *)s;
    sourceLen = len;
    baseAddr = bAddr;
    
    nbLine = sourceLen/16 + ((sourceLen%16) ? 1 : 0);
    
    calcDisplayNeed();
    adjustScrollBar();
}


void QHexEditWidget::calcDisplayNeed(void)
{
    if (source)
    {
	/* We need 80 caracters wide to display */
	maxX = 80 * fontWidth;
	maxY = nbLine * fontHeight;
    }
    else
    {
	maxX = 0;
	maxY = 0;
    }      
}


void QHexEditWidget::adjustScrollBar(void)
{
    if (vScroll == NULL) return;
    
    if (!source)
    {
	hScroll->hide();
	vScroll->hide();
	return;
    }
    
    blockSignals(TRUE);
    
    bool wasHScrollHidden = hScroll->isHidden();
    bool wasVScrollHidden = vScroll->isHidden();
    
    /* Figure out if we will need scroll bars without taking in the fact that the other scroll bar might exists*/
    bool needHScroll = ((vScroll->isHidden() ? 0 : vScroll->width()) + width()) < maxX;
    bool needVScroll = ((hScroll->isHidden() ? 0 : hScroll->height()) + height()) < maxY;
    
    /* This might change our previous decision if we know that the other scroll bar will be shown... */
    if (needVScroll) needHScroll = (width() - (vScroll->isHidden() ? vScroll->width() : 0)) < maxX;
    if (needHScroll) needVScroll = (height() - (hScroll->isHidden() ? hScroll->height() : 0)) < maxY;

    /* Show or hide the scroll bar as needed */
    if (needVScroll && vScroll->isHidden()) vScroll->show();
    if (needHScroll && hScroll->isHidden()) hScroll->show();
    if (!needVScroll && !vScroll->isHidden()) vScroll->hide();
    if (!needHScroll && !hScroll->isHidden()) hScroll->hide();
    
    /* Fix the value of the scroll bars if needed */
    if (needHScroll)
    {
	hScroll->setMinValue(0);
	hScroll->setMaxValue((maxX - width()) / fontWidth + 1);
	hScroll->setValue(x / fontWidth);
	hScroll->setLineStep(1);
	hScroll->setPageStep(width() / fontWidth);
    }
    else
    {
	if (!wasHScrollHidden)
	{
	    x = 0;
	    hScroll->setValue(0);
	}
    }
    
    if (needVScroll)
    {
	vScroll->setMinValue(0);
	vScroll->setMaxValue((maxY - height()) / fontHeight + 1);
	vScroll->setValue(y / fontHeight);
	vScroll->setLineStep(1);
	vScroll->setPageStep(height() / fontHeight);
    }
    else
    {	
	if (!wasVScrollHidden)
	{
	    y = 0;
	    vScroll->setValue(0);
	}
    }
    
    blockSignals(FALSE);
}


void QHexEditWidget::scrollChanged(int)
{
    if (hScroll->isHidden()) x = 0;
    else x = hScroll->value() * fontWidth;
    
    if (vScroll->isHidden()) y = 0;
    else y = vScroll->value() * fontHeight;
    
    paintEvent(NULL);
}


void QHexEditWidget::wheelEvent(QWheelEvent *ev)
{
    /* This will only works with vertical mouse wheel. 
       This will move the content of the widget up or down by ~12% */
    int oldY = y;
    
    if (!source || !vScroll) return;
    if (vScroll->isHidden()) return;
    
    /* Make the 12% move */
    y -= (height() / 12) * (ev->delta() / 120);
    
    /* If 12% of the dump < 1 line, then we will at least move by one line... */
    if (abs(y - oldY) < fontHeight) y = oldY - ((ev->delta() < 0) ? -1 : 1) * fontHeight;
    
    validateY();
    
    /* Update the scroll bar and the widget of there is an actual move done */
    if (oldY != y)
    {
	blockSignals(TRUE);
	vScroll->setValue(y / fontHeight);
	blockSignals(FALSE);
	paintEvent(NULL);
    } 
}


void QHexEditWidget::updateFontCache(void)
{
    /* TO DO : Make the font cache static, i.e. share this font cache with all instance of QHexEditWidget */
    unsigned int i;
    char s[2];

    QPainter paint;
    
    QRect rect(0, 0, fontWidth, fontHeight);
    s[1] = 0;
            
    for (i = 0; i < 256; i++)
    {
	s[0] = i & 0xFF;
       
	/* Generate the black font */
	fontCacheBlack[i] = QPixmap(fontWidth, fontHeight);
	fontCacheBlack[i].fill(Qt::white);
    
	paint.begin(&fontCacheBlack[i], this);
	paint.setFont(font);
	paint.setPen(Qt::black);	
	paint.drawText(rect, Qt::AlignLeft, s);	
	paint.end();
	
	
	/* Generate the blue font */
	fontCacheBlue[i] = QPixmap(fontWidth, fontHeight);
	fontCacheBlue[i].fill(Qt::white);
       
	paint.begin(&fontCacheBlue[i], this);
	paint.setFont(font);
	paint.setPen(Qt::blue);	
	paint.drawText(rect, Qt::AlignLeft, s);	
	paint.end();
    }
}


unsigned int QHexEditWidget::addHighlight(unsigned int from, unsigned int to, QColor color, RasterOp rop)
{
    t_HighlightBlock hi;
    list<t_HighlightBlock>::iterator it;
    
    /* Some sanity check */
    if (!source) return 0;
    if (sourceLen == 0) return 0;
    if (from > sourceLen) from = sourceLen - 1;
    if (to > sourceLen) to = sourceLen - 1;
    
    /* Fill the struct hightlight*/
    hi.start = from;
    hi.end = to;
    hi.no = nextHiNo;
    hi.color = color;
    hi.rop = rop;
    
    /* We will insert at some point into the list, we will keep the list in order of increasing "start" value */
    if (!hiList.size()) hiList.push_back(hi);
    else
    {
	for (it = hiList.begin(); it != hiList.end(); it++)
	{
	    if ((*it).start <= from)
	    {
		hiList.insert(it, hi);
		break;
	    }
	}
	if (it == hiList.end()) hiList.push_back(hi);
    }
    
    
    /* TO DO: figure out if the new hightlight is on the screen and if it is, call paintEvent */
    paintEvent(NULL);
    
    return nextHiNo++;
}



void QHexEditWidget::removeHighlight(unsigned int no)
{
    list<t_HighlightBlock>::iterator it;
    
    if (!hiList.size()) return;
    
    for(it = hiList.begin(); it != hiList.end(); it++)
    {
	if ((*it).no == no) { hiList.erase(it); break; }
    }
    
    /* TO DO: figure out if the removed hightlight is on the screen and if it is, call paintEvent */
    paintEvent(NULL);
}


void QHexEditWidget::mouseReleaseEvent(QMouseEvent *ev)
{
    if (ev->button() != LeftButton) return;
    
    unsigned int res = 0;
    
    int lineMin = y / fontHeight;
    int lineMax = (y + height()) /  fontHeight + 1;
    int lineAt = (y + ev->y()) / fontHeight;
    
    if (lineAt >= lineMin && lineAt <= lineMax)
    {
	/* TO DO: let the user click on the ASCII dump */
	/* TO DO: remove the blank space that the user can click in the hex dump */
	
	/* User clicked on the hex dump */
	if (x + ev->x() >= fontWidth * 12 && x + ev->x() <= fontWidth * 61)
	{
	    res += ((x + ev->x() - fontWidth*12) / 16);
	}
	else return;
	    	    
	res += ((unsigned int)lineAt * 16);
	
	if (res < sourceLen) emit leftClick(res);	
    }   
}


void QHexEditWidget::jumpTo(unsigned int to)
{
    unsigned int firstDisplayed = (y / fontHeight) * 16;
    unsigned lastDisplayed = ((y + height()) / fontHeight) * 16 - 1;
    
    /* Only jump if the data is not currently shown on the screen */
    if (to < firstDisplayed || to > lastDisplayed)
    {
	/* Center the data vertically */
	y = ((to / 16)*fontHeight) - (height() / 2);
	validateY();
	
	/* TO DO: Center the data horizontally */
	
	/* Update the scroll bar and redraw the widget */
	blockSignals(TRUE);
	vScroll->setValue(y / fontHeight);
	blockSignals(FALSE);
	paintEvent(NULL);
    }
}


void QHexEditWidget::validateY(void)
{
    if (y < 0) y = 0;
    
    /* Makes y a factor of fontHeight */
    if ((y % fontHeight) >= fontHeight/2) y += fontHeight;
    y -= (y % fontHeight);
    
    /* Check the upper bound limit */
    if (y > (maxY - height() + fontHeight)) y = ((maxY - height()) / fontHeight + 1) * fontHeight;
}


void QHexEditWidget::blockPaintEvent(bool v)
{
    if (v)
    {
	blockRedraw = true;
    }
    else
    {
	blockRedraw = false;
	if (needRedraw) paintEvent(NULL);
	needRedraw = false;
    }
}
