#include <qpainter.h>
#include <qdragobject.h>

#include "qasmeditwidget.h"

#define CODE_INDENT 4

/* Very useful function that will return a pixmap from the image collection */
QPixmap QAsmEditWidget::lookupPixmap ( const QString name )
{
    const QMimeSource* mime_source = QMimeSourceFactory::defaultFactory()->data( name );
    
    if ( mime_source == 0 ) return QPixmap();
    
    QPixmap pixmap;
    QImageDrag::decode( mime_source, pixmap );
    return pixmap;
}


QAsmEditWidget::QAsmEditWidget(BastardObject *bo, QWidget *parent, const char *name = 0) : QFrame(parent, name)
{
    bastard = bo;
    
    vScroll = NULL;
    hScroll = NULL;
    
    init();
}


QAsmEditWidget::QAsmEditWidget(BastardObject *bo, QScrollBar *v, QScrollBar *h, QWidget *parent, const char *name = 0) : QFrame(parent, name)
{
    bastard = bo;
          
    vScroll = v;
    hScroll = h;

    connect(vScroll, SIGNAL(valueChanged(int)), this, SLOT(scrollChanged(int)));
    connect(hScroll, SIGNAL(valueChanged(int)), this, SLOT(scrollChanged(int)));
    
    init();
}


void QAsmEditWidget::init(void)
{    
    src = NULL;
    
    arrow = -1;
    needUpdateArrow = false;
    arrowPixmap = lookupPixmap("greenarrow.png");
    
    highlight = -1;
    needUpdateHighlight = false;
    
    needUpdateJumpTo = false;
    
    setSourceTicketId = 0;
    
    connect(bastard, SIGNAL(readyNewTicket(unsigned int, void *)), 
	    this, SLOT(slotTicketRecv(unsigned int, void *)));
    connect(bastard, SIGNAL(codeCommentChanged(unsigned int, char *)),
	    this, SLOT(commentChanged(unsigned int, char *)));
        
    font = QFont("Courier");
    QFontMetrics fontInfo = QFontMetrics(font);
    fontWidth = fontInfo.maxWidth();
    fontHeight = fontInfo.lineSpacing();
    
    x = 0;
    y = 0;
        
    calcDisplayNeed();
    adjustScrollBar();
    updateFontCache();
    
    setBackgroundMode(PaletteBase);
}


void QAsmEditWidget::deleteSrc(void)
{
    unsigned int i;
    
    if (!src) return;
    
    if (src->size())
    {
	/* We must delete all the allocated memory via new, we must do this on all char * stuff */
	for (i = 0; i < src->size(); i++)
	{
	    if ((*src)[i].label) delete [] (*src)[i].label;
	    if ((*src)[i].comment) delete [] (*src)[i].comment;
	    if ((*src)[i].instr) delete [] (*src)[i].instr;
	}
    }
    
    delete src;
    
    src = NULL;
}


QAsmEditWidget::~QAsmEditWidget()
{
    deleteSrc();
}


void QAsmEditWidget::paintEvent(QPaintEvent *)
{
    int i, lineMin, lineMax;
    
    /* TO DO: We should keep the whole QPixamp generated from frame to frame,
       while scrolling we can only have to update 10% of the QPixmap make some big optimization */
        
    /* We will draw in a pixmap and later we will blit on the widget to remove flickering */
    QPixmap pm(size());
    pm.fill(); // Set it to a white image
    
    QPainter paint;
    paint.begin(&pm, this);

    /* We are waiting for the data from the bastard thread */
    if (!src)
    {
	drawString(&pm, 0, 0, "Please wait...", -1);
    }
    else
    {
	/* Determine which lines that will actually being printed on the widget */
	lineMin = y / fontHeight;
	lineMax = (y + height()) /  fontHeight + 2;
	if ((unsigned int)lineMax > src->size()) lineMax = src->size();
	
	for (i = lineMin; i < lineMax; i++)
	{
	    /* Line that contains only a label */
	    if ((*src)[i].label) drawString(&pm, 0 - x, (i - lineMin) * fontHeight, (*src)[i].label, -1);
	    else	/* Normal code line, always without a label */
	    {
		drawString(&pm, fontWidth * CODE_INDENT - x, (i - lineMin) * fontHeight, (*src)[i].instr, -1);
		drawString(&pm, fontWidth * (CODE_INDENT + maxLenInstr + 1) - x, (i - lineMin) * fontHeight, (*src)[i].comment, -1);
	    }
	}
    }
    
    /* Draw the green arrow if necessary */
    if (arrow != -1)
    {
	bitBlt(&pm, 0, arrow * fontHeight - y - (arrowPixmap.height()/2 - fontHeight/2), &arrowPixmap);
    }
    
    /* Make the active line highlight if necessary */
    if (highlight != -1)
    {
	paint.setRasterOp(Qt::AndROP);
	paint.setBrush(QBrush(QColor(205, 234, 197)));
	paint.setPen(Qt::NoPen);
	
	paint.drawRect(0, fontHeight * highlight - y, width(), fontHeight);
    }
    
    /* Copy the pixmap on the widget */
    paint.end();
    bitBlt(this, 0, 0, &pm);
}


/* If maxLen is -1, then the whole string is printed */
/* If maxLen > 0, then up to maxLen character are printed (excluding the [...]) */
void QAsmEditWidget::drawString(QPixmap *pm, int deviceX, int deviceY, char *s, int maxLen)
{
    char *tooLong = "[...]";
    char *strEnd;
    int count;
    bool displayTooLong = false;
    
    if (!s || maxLen < -1) return;
    
    /* Find the end of the string s or stop at maxLen */
    count = 0;
    strEnd = s;    
    while (1)
    {
	if (*strEnd == 0) break;
	if (maxLen != -1 && count == maxLen)
	{
	    displayTooLong = true;
	    break;
	}
	strEnd++;
	count++;
    }
    
    /* Display the actual string */
    while (s != strEnd)
    {
	bitBlt(pm, deviceX, deviceY, &fontCacheBlack[*((unsigned char *)s)]);
	deviceX += fontWidth;
	s++;
    }
    
    /* Display the [...] if needed */
    if (displayTooLong) drawString(pm, deviceX, deviceY, tooLong, -1);    
}



void QAsmEditWidget::setSourceSection(struct section *s)
{
    setSourceTicketId = bastard->getCodeSection(s);
}

void QAsmEditWidget::setSourceFunction(BastardFunction *f)
{
    setSourceTicketId = bastard->getCodeFunction(f);
}


void QAsmEditWidget::slotTicketRecv(unsigned int id, void *msg)
{
    if (id == setSourceTicketId)
    {
	displaySource((vector<t_CodeDump> *)msg);
	setSourceTicketId = 0;
    }
}

void QAsmEditWidget::displaySource(vector<t_CodeDump> *c)
{
    unsigned int i;
    
    deleteSrc();
    src = c;
    
    if (!c->size()) return;
    
    maxLenInstr = 0;
    maxLenComment = 0;
    
    /* Figure out the longest instr and comment */
    for (i = 0; i < c->size(); i++)
    {
	if ((*c)[i].instr)
	    if (strlen((*c)[i].instr) > (unsigned int)maxLenInstr) maxLenInstr = strlen((*c)[i].instr);
	
	if ((*c)[i].comment)	
	    if (strlen((*c)[i].comment) > (unsigned int)maxLenComment) maxLenComment = strlen((*c)[i].comment);
    }
        
    calcDisplayNeed();
    adjustScrollBar();
    
    if (needUpdateArrow) setArrow(arrowRVA);
    if (needUpdateHighlight) setHighlight(highlightRVA);
    if (needUpdateJumpTo) jumpTo(jumpToRVA);
    
    paintEvent(NULL);
}


void QAsmEditWidget::updateFontCache(void)
{
    /* TO DO : Make the font cache static, i.e. share this font cache with all instance of QAsmEditWidget */
    unsigned int i;
    char s[2];

    QPainter paint;
    
    QRect rect(0, 0, fontWidth, fontHeight);
    s[1] = 0;
            
    for (i = 0; i < 256; i++)
    {
	s[0] = i & 0xFF;
       
	/* Generate the black font */
	fontCacheBlack[i] = QPixmap(fontWidth, fontHeight);
	fontCacheBlack[i].fill(Qt::white);
    
	paint.begin(&fontCacheBlack[i], this);
	paint.setFont(font);
	paint.setPen(Qt::black);	
	paint.drawText(rect, Qt::AlignLeft, s);	
	paint.end();
    }
}


void QAsmEditWidget::calcDisplayNeed(void)
{
    if (src)
    {
	maxX = (CODE_INDENT + maxLenInstr + maxLenComment + 1) * fontWidth;
	maxY = src->size() * fontHeight;
    }
    else
    {
	maxX = 14 * fontWidth;	// strlen("Please wait...")
	maxY = fontHeight;
    }      
}


void QAsmEditWidget::resizeEvent(QResizeEvent *)
{
    adjustScrollBar();
}


void QAsmEditWidget::adjustScrollBar(void)
{
    if (vScroll == NULL) return;
    
    if (!src)
    {
	hScroll->hide();
	vScroll->hide();
	return;
    }
    
    blockSignals(TRUE);
    
    bool wasHScrollHidden = hScroll->isHidden();
    bool wasVScrollHidden = vScroll->isHidden();
    
    /* Figure out if we will need scroll bars without taking in the fact that the other scroll bar might exists*/
    bool needHScroll = ((vScroll->isHidden() ? 0 : vScroll->width()) + width()) < maxX;
    bool needVScroll = ((hScroll->isHidden() ? 0 : hScroll->height()) + height()) < maxY;
    
    /* This might change our previous decision if we know that the other scroll bar will be shown... */
    if (needVScroll) needHScroll = (width() - (vScroll->isHidden() ? vScroll->width() : 0)) < maxX;
    if (needHScroll) needVScroll = (height() - (hScroll->isHidden() ? hScroll->height() : 0)) < maxY;

    /* Show or hide the scroll bar as needed */
    if (needVScroll && vScroll->isHidden()) vScroll->show();
    if (needHScroll && hScroll->isHidden()) hScroll->show();
    if (!needVScroll && !vScroll->isHidden()) vScroll->hide();
    if (!needHScroll && !hScroll->isHidden()) hScroll->hide();
    
    /* Fix the value of the scroll bars if needed */
    if (needHScroll)
    {
	hScroll->setMinValue(0);
	hScroll->setMaxValue((maxX - width()) / fontWidth + 1);
	hScroll->setValue(x / fontWidth);
	hScroll->setLineStep(1);
	hScroll->setPageStep(width() / fontWidth);
    }
    else
    {
	if (!wasHScrollHidden)
	{
	    x = 0;
	    hScroll->setValue(0);
	}
    }
    
    if (needVScroll)
    {
	vScroll->setMinValue(0);
	vScroll->setMaxValue((maxY - height()) / fontHeight + 1);
	vScroll->setValue(y / fontHeight);
	vScroll->setLineStep(1);
	vScroll->setPageStep(height() / fontHeight);
    }
    else
    {	
	if (!wasVScrollHidden)
	{
	    y = 0;
	    vScroll->setValue(0);
	}
    }
    
    blockSignals(FALSE);
}


void QAsmEditWidget::scrollChanged(int)
{
    if (hScroll->isHidden()) x = 0;
    else x = hScroll->value() * fontWidth;
    
    /* The next line fix a strange bug, hScroll->setMaxValue() from the adjustScrollBar method will call
       this function even if I did block signals, when this method get called, hScroll->value() return a bogus
       value that is fixed with the following line.*/
    if (x < 0) x = 0;
    
    if (vScroll->isHidden()) y = 0;
    else y = vScroll->value() * fontHeight;
    
    paintEvent(NULL);
}


void QAsmEditWidget::wheelEvent(QWheelEvent *ev)
{
    /* This will only works with vertical mouse wheel. 
       This will move the content of the widget up or down by ~12% */
    int oldY = y;
    
    if (!src || !vScroll) return;
    if (vScroll->isHidden()) return;
    
    /* Make the 12% move */
    y -= (height() / 12) * (ev->delta() / 120);
    
    /* If 12% of the dump < 1 line, then we will at least move by one line... */
    if (abs(y - oldY) < fontHeight) y = oldY - ((ev->delta() < 0) ? -1 : 1) * fontHeight;
    
    validateY();
    
    /* Update the scroll bar and the widget if there is an actual move done */
    if (oldY != y)
    {
	blockSignals(TRUE);
	vScroll->setValue(y / fontHeight);
	blockSignals(FALSE);
	paintEvent(NULL);
    } 
}


void QAsmEditWidget::setArrow(unsigned int rva)
{
    unsigned int i;
    
    arrowRVA = rva;
    
    if (!src)
    {
	needUpdateArrow = true;
	return;
    }
    else
    {
	needUpdateArrow = false;
    }
    
    if (!src->size()) return;
    
    /* src vector is sorted by rva */
    for (i = 0; i < src->size(); i++)
    {
	if ((*src)[i].rva == rva)
	{
	    /* Put it on the next line if it's a label line */
	    if ((*src)[i].label != NULL) arrow = i + 1;
	    else arrow = i;
	}
    }
    
    /* TO DO: Only paint when it is necessary */
    paintEvent(NULL);
}


void QAsmEditWidget::setHighlight(unsigned int rva)
{
    unsigned int i;
    
    highlightRVA = rva;
    
    if (!src)
    {
	needUpdateHighlight = true;
	return;
    }
    else
    {
	needUpdateHighlight = false;
    }
    
    if (!src->size()) return;
    
    /* src vector is sorted by rva */
    for (i = 0; i < src->size(); i++)
    {
	if ((*src)[i].rva == rva)
	{
	    /* Put it on the next line if it's a label line */
	    if ((*src)[i].label != NULL) highlight = i + 1;
	    else highlight = i;
	    
	    /* Emit the signal */
	    emit clickOnRVA(rva, NoButton);
	    
	    break;
	}
    }
    
    /* TO DO: Only paint when it is necessary */
    paintEvent(NULL);
}


void QAsmEditWidget::jumpTo(unsigned int rva)
{
    unsigned int i;
    int oldY, oldX;
    
    if (!src)
    {
	needUpdateJumpTo = true;
	return;
    }
    else
    {
	needUpdateJumpTo = false;
    }
    
    if (!src->size()) return;
    if (!vScroll) return;
    
    oldY = y;
    oldX = x;
    
    /* Find the line where we must jump to */
    for (i = 0; i < src->size(); i++)
    {
	if ((*src)[i].rva == rva) break;
    }
    
    /* We didn't find the specified RVA, dont do nothing then */
    if (i == src->size()) return;
    
    if (maxY <= height())
    {
	y = 0;
	x = 0;
    }
    else
    {
	y = i * fontHeight;
	x = 0;
	
	validateY();
    }
    
    /* Update the scroll bar and the widget of there is an actual move done */
    if (oldY != y || oldX != x)
    {
	blockSignals(TRUE);
	vScroll->setValue(y / fontHeight);
	blockSignals(FALSE);
	paintEvent(NULL);
    } 
 
}


void QAsmEditWidget::mouseReleaseEvent(QMouseEvent *ev)
{
    if (ev->button() != LeftButton) return;
    if (!src) return;
    
    int userHighlight = (y + ev->y()) / fontHeight;
    
    /* Check if the user did click on a line, not at the bottom of the widget where it might contain nothing */
    if ((unsigned int)(y + ev->y()) < fontHeight * src->size())
    {
	/* Do not let the user select a label line */
	if (/*userHighlight != highlight &&*/ (*src)[userHighlight].label == NULL)
	{
	    highlight = (y + ev->y()) / fontHeight;
	    paintEvent(NULL);
	    
	    emit clickOnRVA((*src)[highlight].rva, ev->state());
	}
    }
}


void QAsmEditWidget::commentChanged(unsigned int rva, char *cmt)
{
    unsigned int i;
    
    if (!src) return;
    if (!src->size()) return;
    
    for (i = 0; i < src->size(); i++)
    {
	if ((*src)[i].rva == rva)
	{
	    if ((*src)[i].comment != NULL) delete [] (*src)[i].comment;
	    
	    (*src)[i].comment = new char[strlen(cmt)+2];
	    /* TO DO: Accept other form of comment other than ; */
	    (*src)[i].comment[0] = ';';
	    strcpy(&((*src)[i].comment[1]), cmt);
	    
	    /* Check if the new comment is longer than the previous longuest comment */
	    if (maxLenComment < (int)strlen((*src)[i].comment))
	    {
		maxLenComment = strlen((*src)[i].comment);
		calcDisplayNeed();
		adjustScrollBar();
		paintEvent(NULL);
	    }
	    else
	    {
		/* TO DO: Only refresh the widget if the comment is visible */
		paintEvent(NULL);
	    }
	}
    }
}


void QAsmEditWidget::validateY(void)
{
    if (y < 0) y = 0;
    
    /* Makes y a factor of fontHeight */
    if ((y % fontHeight) >= fontHeight/2) y += fontHeight;
    y -= (y % fontHeight);
    
    /* Check the upper bound limit */
    if (y > (maxY - height() + fontHeight)) y = ((maxY - height()) / fontHeight + 1) * fontHeight;
}
