#include "signatures.h"
#include "sign_tree.h"

struct EXT__PLUGIN *settings;
extern struct EXT__ARCH *ext_arch;

struct SIGN_SYM_LIST {
	sign_sym_t *sym;
	struct SIGN_SYM_LIST *prev, *next;
} sign_sym_list = {0};

sign_tree_t *match_tree = NULL, tree = {0};

static int sign_sym_list_free( void );

/* ---------------------------------------------------- EXT_PLUGIN init() */
void ext_plugin_init(void *param)
{
	settings = (struct EXT__PLUGIN *) param;
	return;
}
/* ---------------------------------------------------- EXT_PLUGIN cleanup() */
void ext_plugin_cleanup(void)
{
	/* if symbols have already been generated, delete them */
	if ( sign_sym_list.next ) {
		sign_sym_list_free();
	}
	if ( tree.libs ) {
		;/* free libs */
	}
	if ( match_tree ) {
		sign_tree_free( tree.libs );
	}
	return;
}
/* ---------------------------------------------------- EXT_PLUGIN main() */
int plugin_main(void *param)
{
	return (1);
}

static int sign_sym_list_free( void ) {
	struct SIGN_SYM_LIST *s, *next = sign_sym_list.next;

	s = sign_sym_list.next;
	while ( s ) {
		next = s->next;
		free( s->sym );
		free( s );
		s = next;
	}
	return( 1 );
}
			
static int sign_sym_list_add( sign_sym_t *sym ) {
	struct SIGN_SYM_LIST *s, *prev = sign_sym_list.next;
	
	if ( ! sym )	return(0);
	s = calloc( sizeof(struct SIGN_SYM_LIST), 1 );
	if ( ! s )	return(0);
	s->sym = sym;

	if ( ! sign_sym_list.next ) {
		sign_sym_list.next = s;
	} else {
		while ( prev->next ) {
			prev = prev->next;
		}
		prev->next = s;
		s->prev = prev;
	}
	return(1);
}

/* For now, we just implement Van Emmerik */
/* SIG_VAN_EMMERIK */

/* generate signature for a range of bytes */
static int sign_gen_bytes( sign_sym_t *sym, unsigned char *buf, int buf_len) {
	struct ARCH_INVARIANT inv = {0};
	unsigned char *pos;	/* position in sym->sig */
	int type, cont = 1;

	sym->sig = calloc( SIGN_MAX, 1 );
	if (! sym->sig)	return(0);
	pos = sym->sig;
	sym->sig_len = 0;	/* stop compiler bitching */

	while ( cont && buf_len > 0  && sym->sig_len < SIGN_MAX ) {
		cont = ext_disasm_invariant( buf, buf_len, &inv );

		if (! cont )	continue;
		if ( inv.buf_len > (SIGN_MAX - sym->sig_len) ) {
			cont = 0;
			continue;
		}

		buf += cont;
		buf_len -= cont;
		
		/* append to symbol signature */
		memcpy( pos, inv.buf, inv.buf_len );
		pos += inv.buf_len;
		sym->sig_len += inv.buf_len;

		/* stop at jmp/ret insns */
		type = INS_TYPE(inv.insn_type);
		if ( type == INS_BRANCH ||
		     type == INS_RET ||
		     buf_len <= 0 ) {
			cont = 0;
		}
	}

	return( sym->sig_len );
}

/* generate signature for a function */
static sign_sym_t * sign_gen_func( sign_lib_t *lib, struct function *func, char *name ) {
	struct DISASM_TGT *target;
	unsigned char *buf;
	struct address a;
	struct comment c;
	sign_sym_t *sym = NULL;

	if ( bdb_index_find( ADDRESS_RVA, &func->rva, &a ) ) {
		target = env_get_target();
		buf = target->image + a.pa;
		sym = calloc( sizeof(sign_sym_t), 1 );
		if (! sym) return(NULL);
		sym->name = strdup(name);
		sym->ret = strdup(" ");
		sym->args = strdup(" ");
		/* get comment for name */
		if ( bdb_index_find( COMMENT_ID, &func->comment, &c ) ) {
			sym->comment = strdup(c.text);
		} else {
			sym->comment = strdup(" ");
		}
		sym->lib = lib;	/* not that this matters */
		if ( ! func->size ) {
			func->size = (target->info.size - a.pa) - 1;
		}
		sign_gen_bytes( sym, buf, func->size );
	} 
	
	return( sym );
}

static void sign_func_cb( struct function *f, sign_lib_t *lib ) {
	sign_sym_t *sym;
	struct name n;
	struct import_addr i;
	char buf[PATH_MAX];

	if ( bdb_index_find( IMPORT_ADDR_RVA, &f->rva, &i )	||
	     ! bdb_index_find( NAME_RVA, &f->rva, &n ) ) {
		/* skip imported functions and unnamed functions */
		return;
	}
	
	if ( ! (n.type & NAME_AUTO) ||  n.type & NAME_SYMBOL ) {
		sym = sign_gen_func( lib, f, n.text );
		if ( sym ) {
			sprintf( buf, "%s : %d byte signature generated\n", n.text, 
						sym->sig_len);
			sys_msg( buf );
			sign_sym_list_add( sym );
		}
	} 
	return;
}


/* generate signature for all functions in target */
int sign_gen_target( char *entry_sym ) {
	struct DISASM_TGT *target;
	struct SIGN_SYM_LIST *l;
	sign_lib_t *lib;
	
	target = env_get_target();

	/* generate sign_lib_t for target */
	lib = calloc( sizeof(sign_lib_t), 1 );
	if (! lib ) 	return(sys_set_lasterr(9050));
	/* the tree isn't really important; it is used in matching */
	tree.libs = lib;
	tree.type = SIG_VAN_EMMERIK;
	tree.wc = ext_arch->wc_byte;

	lib->vendor = (target->info.vendor[0])? target->info.vendor : " ";
	lib->model = (target->info.model[0])? target->info.model : " ";
	lib->version =  target->info.version;
	lib->comment = (target->info.comment[0])? target->info.comment : " ";
	lib->name = (target->info.name[0])? target->info.name : " ";
	lib->arch = (target->arch.name[0])? target->arch.name : " ";
	lib->comp = (target->comp.name[0])? target->comp.name : " ";
	lib->lang = (target->lang.name[0])? target->lang.name : " ";
	lib->os = (target->os.name[0])?  target->os.name : " ";

	/* if symbols have already been generated, delete them */
	if ( sign_sym_list.next ) {
		sign_sym_list_free();
	}

	/* reuse list_head->sym as lib */
	sign_sym_list.sym = (sign_sym_t *)lib;

	/* do callback for each function */
	func_foreach( (BAPI_CALLBACK)sign_func_cb, lib );

	/* if _start is defined, make it the entry sym */
	l = sign_sym_list.next;
	while( entry_sym && l ) {
		if ( ! strcmp( l->sym->name, entry_sym ) ) {
			lib->entry = l->sym;
			break;
		}
		l = l->next;
	}

	return(1);
}

static void sign_fprint_sym( FILE *f, sign_sym_t *sym ){
	unsigned char buf[SIGN_MAX+1];
	int i;

	fprintf( f, "%s|%s|%s|%s|%x|", sym->name, sym->ret, 
		sym->args, sym->comment, sym->sig_len);
	for ( i=0; i < sym->sig_len; i++ ) {
		buf[i] = sym->sig[i];	
	}	
	buf[sym->sig_len] = '\n';
	fwrite( buf, sym->sig_len + 1, 1, f );
	return;
}

int sign_write_file( FILE *f ) {
	struct SIGN_SYM_LIST *l;
	sign_lib_t *lib;
	sign_sym_t *sym;
	char buf[PATH_MAX];

	lib = (sign_lib_t *) sign_sym_list.sym;
	if ( ! lib || ! sign_sym_list.next ) {
		fprintf(stderr, "No signatures to write to file\n");
		return( 0 );
	}
	fseek( f, 0, SEEK_SET);
	/* write header */
	fprintf( f, "%s|%2.2f|%02X|%02X\n", SIGN_MAGIC, SIGN_VER, tree.wc, tree.type );

	/* write library info */
	fprintf( f, "%s|%s|%s|%2.2f|%s\n", lib->name, lib->vendor, lib->model,
			lib->version, lib->comment );
	fprintf( f, "%s|%s|%s|%s\n", lib->arch, lib->comp, lib->lang, lib->os );

	/* write entry point */
	
	/* write sigs */
	l = sign_sym_list.next;
	while ( l ) {
		sign_fprint_sym( f, l->sym );
		l = l->next;
	}

	return(1);
}

int sign_load_file( FILE *f ) {
	sign_lib_t *lib;
	sign_sym_t *sym;
	char line[256], entry_sym[128];
	float ver;
	int len, wc, type, cont = 1;
	char junk, *buf, *buf2, *buf3, *buf4;

	buf = malloc( (SIGN_MAX > PATH_MAX) ? SIGN_MAX : PATH_MAX );
	buf2 = malloc( PATH_MAX );
	buf3 = malloc( PATH_MAX );
	buf4 = malloc( PATH_MAX );

	/* get header */
	fseek( f, 0, SEEK_SET);
	fgets( line, 256, f );
	/* verify header */
	buf[0] = buf2[0] = buf3[0] = buf4[0] = '\0';
	sscanf( line, "%[^|]|%f|%X|%X\n", buf, &ver, &wc, &type );

	/* if ! tree, create new tree */
	if ( ! match_tree ) {
		match_tree = sign_tree_new( type, wc );
		if ( ! match_tree ) {
			fprintf( stderr, "SIGNATURES: Unable to create sig tree!\n");
			return(0);
		}
	}

	/* get library info */
	lib = calloc( sizeof(sign_lib_t), 1 );
	if ( ! lib ) 	return(0);

	fgets( line, 256, f );
	buf[0] = buf2[0] = buf3[0] = buf4[0] = '\0';
	sscanf( line, "%[^|]|%[^|]|%[^|]|%f|%s\n", buf, buf2, buf3, 
			&lib->version, buf4 );
	if ( buf[0] ) lib->name = strdup(buf);
	if ( buf2[0] ) lib->vendor = strdup(buf2);
	if ( buf3[0] ) lib->model = strdup(buf3);
	if ( buf4[0] ) lib->comment = strdup(buf4);
	snprintf( line, 256, "Library %s %s version %2.2f  %s", lib->name, 
			lib->model ? lib->model : " ", lib->version, 
			lib->vendor ? lib->vendor : " " );
	sys_msg( line );
	snprintf( line, 256, "(%s)\n", lib->comment ? lib->comment : "no cmt" );
	sys_msg( line );

	fgets( line, 256, f );
	buf[0] = buf2[0] = buf3[0] = buf4[0] = '\0';
	sscanf( line, "%[^|]|%[^|]|%[^|]|%s\n", buf, buf2, buf3, buf4 );
	if ( buf[0] ) lib->arch = strdup(buf);
	if ( buf2[0] ) lib->comp = strdup(buf2);
	if ( buf3[0] ) lib->lang = strdup(buf3);
	if ( buf4[0] ) lib->os = strdup(buf4);
	sign_tree_add_lib( match_tree, lib );

	/* foreach signature, insert into tree */
	while ( cont && cont != EOF ) {
		buf[0] = buf2[0] = buf3[0] = buf4[0] = '\0';
		/* get symbol info */
		cont = fscanf( f, "%[^|]|%[^|]|%[^|]|%[^|]|%x|", buf, buf2, 
				   buf3, buf4, &len);
		if ( buf[0] && len ) {
			/* we have a symbol and a signature ! */
			sym = calloc( sizeof(sign_sym_t), 1 );
			if (! sym ) {
				cont = 0;
				continue;
			}
			
			if ( buf[0] ) sym->name = strdup(buf);
			if ( buf2[0] ) sym->ret = strdup(buf2);
			if ( buf3[0] ) sym->args = strdup(buf3);
			if ( buf4[0] ) sym->comment = strdup(buf4);

			sym->sig_len = len;

			/* get signature */
			sym->sig = malloc(sym->sig_len);
			fread( sym->sig, len, 1, f );
			fread( &junk, 1, 1, f );	/* strip \n */

			/* add to tree */
			/* printf("SYM insert: %s len %d\n", sym->name, len); */
			sign_tree_ins( match_tree, sym );
		}
		/* continue to next symbol */
	}

	/* sign_tree_print( match_tree, stdout ); */

	free(buf);
	free(buf2);
	free(buf3);
	free(buf4);
	return(1);
}

static sign_sym_t * sign_match_bytes( void *tree, unsigned char *buf, int buf_len ) {
	unsigned char *sig;
	int sig_len;
	sign_tree_t *t = tree;
	if ( t->type == SIG_VAN_EMMERIK ) {
		sig = buf;
		sig_len = buf_len;
	} else {
		/* for now... */
		return(NULL);
		/* later, generate proper non-van-emm signature in 'buf' */
	}
	return( sign_tree_match( tree, sig, sig_len ) );
}

static int sign_update_func( struct function *func, sign_sym_t *sym ) {
	if ( ! func || ! sym )	return(0);

	/* printf("MATCH: rva %X is %s\n", func->rva, sym->name ); */
	/* rename function */
	func_set_name( func->rva, sym->name );

	if ( sym->ret ) {
		;	/* set function type */
	}
	if ( sym->args ) {
		;	/* set function args */
	}
	if ( sym->comment ) {
		; 	/* set function comment */
	}
	return(1);
}

int sign_match_func( struct function *func ) {
	int buf_len;
	sign_sym_t *sym;
	unsigned char *buf;
	struct address a;
	struct DISASM_TGT *target = env_get_target();
	
	if ( ! match_tree ) {
		fprintf( stderr, "No signature files loaded!\n");
		return(0);
	}

	if ( bdb_index_find( ADDRESS_RVA, &func->rva, &a ) ){
		buf = target->image + a.pa;
		buf_len = target->info.size - a.pa;

		sym = sign_match_bytes( match_tree, buf, buf_len );

		if ( sym ) {
			sign_update_func( func, sym );
			return(1);
		}
	}
	return(0);
}

int sign_match_sec( struct section *sec ) {
	int buf_len;
	sign_sym_t *sym;
	unsigned char *buf;
	
	if ( ! match_tree ) {
		fprintf( stderr, "No signature files loaded!\n");
	}

	/* scan for bytes in section */
	return(0);
}

/* SIG_VAN_EM_PLUS */
/* SIG_MD5 */
/* SIG_INSN_TYPES */
