#include "sign_tree.h"

void * sign_tree_new( char type, char wc ) {
	sign_tree_t *tree;
	tree = calloc( sizeof(sign_tree_t), 1 );
	if ( tree ) {
		tree->wc = wc;
		tree->type = type;
	}
	return( tree );
}

/* notice the sym_copy and sym_free are here: this not reusable tree :P */
static int sign_sym_free( sign_sym_t *sym ) {
	if (! sym )		return(0);
	if ( sym->name )		free( sym->name );
	if ( sym->ret )		free( sym->ret );
	if ( sym->args )		free( sym->args );
	if ( sym->comment )	free( sym->comment );
	if ( sym->sig )		free( sym->sig );
	free( sym );
	return(1);
}

static int sign_tree_node_free( sign_node_t * node ) {
	sign_leaf_t *leaf;
	if ( ! node ) 	return(0);
	if ( node->type == SIGN_TREE_NODE ) {
		if ( node->mid ) 	sign_tree_node_free( node->mid );
		if ( node->left )	sign_tree_node_free( node->left );
		if ( node->right )	sign_tree_node_free( node->right );
	}
	if ( node->type == SIGN_TREE_LEAF ) {
		leaf = (sign_leaf_t *)node;
		sign_sym_free( leaf->sym );
	}
	free( node );
	return(1);
}

int sign_tree_free( void *tree ) {
	sign_tree_t *t = (sign_tree_t *)tree;
	sign_lib_t *l;
	if ( ! t ) 	return(0);

	l = t->libs;
	while( l ) {
		t->libs = l->next;
		free(l);
		l = t->libs;
	}

	return( sign_tree_node_free( t->root ) );
}

/* note: in this function 'i' is returned as a way of indicating which buffer
 * element failed to match -- i.e., where to split the buffer or fail the
 * match. 'i' can be pos or neg to indicate if buf was < > node->data. If
 * element 0 fails to match, however, it cannot be marked +/- ... therefore
 * i is incremented before being returned and then decremented in the
 * 'insert' function. */
static int sign_tree_node_cmp(sign_node_t *node, unsigned char *buf, int len, 
	unsigned char wc){
	int i, lim;

	if ( ! node || ! buf || node->type != SIGN_TREE_NODE ) {
		return( 0 );
	}

	/* max # of bytes to check -- extent of buf/node->data */
	lim = ( len < node->len ) ? len : node->len;
	for ( i = 0; i < lim; i++ ){
		/* debug code for later printf("%X == %X ? (%d bytes) wc=%02x\n", 
			buf[i], node->data[i], lim, wc); */
		if ( wc && node->data[i] == wc ) 
			/* doesn't matter what buf[i] is, node[i] is a wildcard */
			continue;
		if ( buf[i] < node->data[i] ) {
			return( -(++i) );	/* go to left child of node */
		} else if ( buf[i] > node->data[i] ) {
			return( ++i );	/* go to right child of node */
		}
		/* so far, we have a match */
	}
	/* make sure buf is not shorter than node->data */
	if ( node->len > len ){
		return( -(++i) );
	}
	/* okay, everything matches */
	return(0);
}

static sign_sym_t * sign_sym_new( sign_sym_t *sym ) {
	sign_sym_t *s;
	if (! sym || ! sym->name )	return(0);
	s = calloc( sizeof(sign_sym_t), 1);
	if (! s )	return(0);
	s->name = strdup(sym->name);
	s->ret = strdup(sym->ret);
	s->args = strdup(sym->args);
	s->comment = strdup(sym->comment);
	s->sig = sym->sig;	/* sig is already allocated */

	return( s );
}

static sign_node_t * sign_tree_node_new( char type, void *data, short len ) {
	sign_node_t *node;
	sign_leaf_t *leaf;

	if ( type == SIGN_TREE_NODE ) {
		node = calloc( sizeof(sign_node_t), 1 );
		if (! node )	return(NULL);
		node->data = data;
	} else {
		leaf = calloc( sizeof(sign_leaf_t), 1 );
		if (! leaf )	return(NULL);
		/* we don't need to do this actually :)
		leaf->sym = sign_sym_new(data); */
		leaf->sym = (sign_sym_t *) data;
		node = (sign_node_t *) leaf;
	}
	node->type = type;
	node->len = len;
	return( node );
}

static int sign_tree_node_branch( sign_node_t *node, unsigned char *buf, 
				int len, sign_sym_t *sym, int diff, char left ) {
	sign_node_t *n;

	if (! node || ! buf || ! sym || ! diff ) return(0);

	/* diff is the index where the bytes differ */
	if ( diff < 0 ) 	diff *= -1;
	/* is buf shorter than node? */
	if ( len < node->len ) {
		/* do not add : ambiguous node */
		return(0);
	} else {
		/* split 'node' at the point where the bytes differ */
		n = sign_tree_node_new( SIGN_TREE_NODE, &node->data[diff], 
						node->len - diff );
		node->len = diff;
		if ( node->mid ) {
			n->mid = node->mid;
		}
		node->mid = n;
		/* switch to new node */
		node = n;
	}

	n = sign_tree_node_new( SIGN_TREE_NODE, &buf[diff], len - diff );
	n->mid = sign_tree_node_new( SIGN_TREE_LEAF, sym, 0 );
	/* create branch for differing bytes in 'buf' */
	if ( left ) 
		node->left = n;
	else 
		node->right = n;
	return(1);
}

int sign_tree_ins( void *tree, sign_sym_t *sym ) {
	sign_node_t *node;
	unsigned char *buf;
	int buf_len, diff;
	sign_tree_t *t = (sign_tree_t *)tree;

	if (! t || ! sym || ! sym->sig )	return(0);
	buf = sym->sig;
	buf_len = sym->sig_len;

	if (! t->root ) {
		t->root = sign_tree_node_new( SIGN_TREE_NODE, buf, buf_len );
		t->root->mid = sign_tree_node_new( SIGN_TREE_LEAF, sym, 0 );
		return(1);
	}

	node = t->root;
	while ( node && node->type != SIGN_TREE_LEAF ) {
		/* note we do not use wildcard when inserting :) */
		diff = sign_tree_node_cmp( node, buf, buf_len, 0 );
		/* remember diff was advanced in cmp function so that
		 * returning index 0 could be neg/pos ... diff-- is the
		 * index of the buffer element that failed to match */
		if ( diff < 0 ) {
			if ( ! node->left ) {
				return( sign_tree_node_branch( node, buf, buf_len, 
							sym, ++diff, 1 ) );
			}
			node = node->left;
		} else if ( diff > 0 ) {
			if ( ! node->right ) {
				return( sign_tree_node_branch( node, buf, buf_len, 
							sym, --diff, 0 ) );
			}
			node = node->right;
		} else {
			if ( node->mid ) {
				/* continue down this trail */
				buf_len -= node->len;
				buf += node->len;
				node = node->mid;
			} else {
				/* weird, this should not happen: match w/o leaf */
				node->mid = sign_tree_node_new( SIGN_TREE_LEAF,
						  		sym, 0 );
				return( 1 );
			}
		}	/* end else if ( diff == 0 ) */
	}	/* end while ( node ) */
	return( 0 );
}

sign_sym_t * sign_tree_match( void *tree, unsigned char *buf, int len ) {
	sign_tree_t *t = (sign_tree_t *)tree;
	sign_node_t *node;
	sign_leaf_t *leaf;
	unsigned char *pos;
	int diff;
	
	if (! t || ! buf )	return( NULL );
	pos = buf;
	node = t->root;
	while ( node ) {
		/* debug code in case it's needed :) 
		printf("matching %02X to (%X) %02X @ %d\n", 
			   buf[0], node, node->data[0], buf-pos );   */
		diff = sign_tree_node_cmp( node, buf, len, t->wc );
		if ( diff > 0 ) {
			/* branch right */
			node = node->right;
		} else if ( diff < 0 ) {
			/* branch left */
			node = node->left;
		} else {
			buf += node->len;
			len -= node->len;
			node = node->mid;
			if ( node ) {
				/* bytes match: continue compare or return sym */
				if ( node->type == SIGN_TREE_LEAF ) {
					/* end of the line: a match */
					leaf = (sign_leaf_t *)node;
					return( leaf->sym );
				}
			}
		}	/* end if ( diff == 0 ) */
	}	/* end while ( node ) */
	return ( NULL );
}

/* note: entry point is set when reading a library */
sign_sym_t * sign_tree_sym_entry( void *tree ) {
	sign_tree_t *t = tree;
	sign_lib_t *l;

	if ( ! tree )	return(NULL);
	l = t->libs;
	while ( l ) {
		if ( l->entry ) {
			return( l->entry );
		}
		l = l->next;
	}
	return(NULL);
}

sign_lib_t * sign_tree_add_lib( void *tree, sign_lib_t *lib ) {
	sign_tree_t *t = tree;
	sign_lib_t *l;

	if ( ! tree || ! lib)	return(NULL);
	l = t->libs;
	if ( ! l ) {
		t->libs = l = calloc( sizeof(sign_lib_t), 1 );
		l = t->libs;
	} else {
		while ( l->next ) {
			l = l->next;
		}
		l->next = calloc( sizeof(sign_lib_t), 1);
		l = l->next;
	}
	if (! l)	return(0);
	memcpy( l, lib, sizeof(sign_lib_t) );
	l->next = NULL;
	return(l);
}

static int sign_tree_node_print( sign_node_t *node, FILE *f, int tab ) {
	sign_sym_t *s;
	int i;

	if (! node )	return(0);
	
	/* print data for tree */
	if ( node->type == SIGN_TREE_LEAF ) {
		s = (sign_sym_t *) node->data;
		fprintf(f, " : LEAF (%s)\n", s->name);
		return(1);
	} else {
		for ( i = 0; i < tab; i++ )	fprintf(f, "\t");
		for ( i = 0; i < node->len; i++ ) {
			fprintf( f, "%02X ", node->data[i]);
		}
		fprintf( f, "\n" );
	}

	if ( node->left ){
		fprintf( f, "Left:\n" );
		sign_tree_node_print( node->left, f, ++tab );
	}
	if ( node->mid ){
		fprintf( f, "Mid:\n" );
		sign_tree_node_print( node->mid, f, ++tab );
	}
	if ( node->right ){
		fprintf( f, "Right:\n" );
		sign_tree_node_print( node->right, f, ++tab );
	}
	return(1);	
}

int sign_tree_print( void *tree, FILE *file ) {
	sign_node_t *node;

	sign_tree_node_print( ((sign_tree_t *)tree)->root, file, 0 );
	return(1);
}

