#include "db_bsql.h"

/* =============================================== POINTLESS CODE */
/* generic tree node operator: in case 'op' doesn't get initialized */

char * bsql_tree_gennodeop ( struct TREENODE *n, void *data ) {
	return(NULL);
}

void bsql_tree_destroy( struct TREENODE *n ) {
	if (! n) return;
	if ( n->op != bsql_valref_nodeop ) {
		/* don't try to traverse LEAF nodes :) */
		if ( n->lc ) bsql_tree_destroy( n->lc );
		if ( n->rc ) bsql_tree_destroy( n->rc );
	}
	free(n);
	return;
}


/* =============================================== TOP-LEVEL TREE NODES */
/* note bsql_exec() has to figure out these return values based on top-level
 * node->op addresses :P */

/* this is a top-level select, not a subselect. Its right child is NULL
 * and its left child is a NODE returning a TMP_TABLE. Returns a string
 * with the result of the select delimited according to BSQL->settings. This
 * string MUST be freed */
char * bsql_select_nodeop( struct TREENODE *n, void *data ){
	struct BSQL_TMPTBL *t;
	char *buf;

	if ( ! n || ! n->lc ) 	return(NULL);
	/* get temptable from sub-select */
	t = n->lc->op( n->lc, NULL );
	if (! t) 	return(NULL);

	buf = bsql_tmptbl_tostr( t );
	bsql_tmptbl_destroy( t );
	return( buf );
}

/* top-level insert. Returns number of items inserted. *
 * 	left node returns a temptable [e.g, is a table_expr or subselect]
 * 	this temptable is a table of items to be inserted.
 * 	right node returns a VAL_T containing the typhoon table ID */
/* TODO: create 2nd-level INSERT to send an empty, right-sized table to ASGN */
int bsql_insert_nodeop( struct TREENODE *n, void *data ){
	struct BSQL_TMPTBL *t;
	struct BSQL_VALREF *v;
	struct db_table *table;

	if ( ! n || ! n->lc || ! n->rc )	return(0);
	t = n->lc->op( n->lc, NULL );
	if (! t) 	return(NULL);
	v = n->rc->op( n->rc, NULL );
	if (! v) {
		bsql_tmptbl_destroy( t );
		return(0);
	}
	table = v->u.table;

	/* insert each row in the temp table */
	row = t->rows;
	while ( row ) {
		/* this is not goin to work until 1) we pass a TMPTBL to ASGN
		 * and 2) db_record_insert() handles sequences automatically */
		//db_record_insert(table->index->id, row->data);
		row = row->next;
	}

	free( v );
	bsql_tmptbl_destroy( t );
	return( 1 );
}

/* top level update. Returns number of items updated *
 * 	left node returns a temptable [e.g, is an update-asgn ]
 * 	this temptable is a table of items to be updated as-is.
 * 	right node returns a VAL_REF containing the typhoon table ID */
int bsql_update_nodeop( struct TREENODE *n, void *data ){
	struct BSQL_TMPTBL *t;
	struct BSQL_VALREF *v;
	struct BSQL_ROW *row;
	struct db_table *table;
	int offset;

	if ( ! n || ! n->lc || ! n->rc )	return(0);
	t = n->lc->op( n->lc, NULL );
	if (! t) 	return(NULL);
	v = n->rc->op( n->rc, NULL );
	if (! v) {
		bsql_tmptbl_destroy( t );
		return(0);
	}

	table = v->u.table;
	offset = bsql_tmptbl_keyoffset( t, table->index->key1->name );

	/* find and update each row in the temp table */
	row = t->rows;
	while ( row ) {
		db_record_update(table->index->id, &row->data[offset], row->data);
		row = row->next;
	}
	
	free( v );
	bsql_tmptbl_destroy( t );
	return( 1 );
}

/* top-level delete. Returns number of items deleted. 
 * 	left node returns a temptable [e.g, is a table_expr or subselect]
 * 	this temptable is a table of items to be deleted.
 * 	right node returns a VAL_T containing the typhoon table ID */
int bsql_delete_nodeop( struct TREENODE *n, void *data ){
	struct BSQL_TMPTBL *t;
	struct BSQL_VALREF *v;
	struct BSQL_ROW *row;
	struct db_table *table;
	int offset;

	if ( ! n || ! n->lc || ! n->rc )	return(0);
	t = n->lc->op( n->lc, NULL );

	if (! t) 	return(NULL);
	v = n->rc->op( n->rc, NULL );
	if (! v) {
		bsql_tmptbl_destroy( t );
		return(0);
	}
	table = v->u.table;
	offset = bsql_tmptbl_keyoffset( t, table->index->key1->name );

	/* find and delete each row in the temp table */
	row = t->rows;
	while ( row ) {
		db_record_delete(table->index->id, &row->data[offset]);
		row = row->next;
	}

	free( v );
	bsql_tmptbl_destroy( t );
	return( 1 );
}




/* ====================================================== SECONDARY NODES */
/* INSERT-ASGN : Left node is an ASGN, right node is a COL_LIST */
/*               Returns table of records to be changed in the DB */
struct BSQL_TMPTBL * bsql_insertasgn_nodeop( struct TREENODE *n, void *data ) {
	struct BSQL_TMPTBL *t == NULL;
	struct BSQL_COL *cols;

	if (! n || ! n->lc || ! n->rc)	return(NULL);

	/* get table of values to insert */
	t = n->lc->op( n->lc, NULL );
	/* list of columns generated from db_table */
	cols = n->rc->op( n->rc, NULL );
	/* create new table mapping values into actual table type */
	/* TODO: create row */

	/* return  updated items for writing to Typhoon DB */
	return( t );
}

/* UPDATE-ASGN : Left node is an ASGN, right node is a SUB_SEL */
/*               Returns table of records to be changed in the DB
 *               Should only be called by top-level UPDATE */
struct BSQL_TMPTBL * bsql_updateasgn_nodeop( struct TREENODE *n, void *data ) {
	struct BSQL_TMPTBL *t;

	if (! n || ! n->lc || ! n->rc)	return(NULL);
	/* get SELECT of items to UPDATE from table */
	t = n->rc->op( n->rc, NULL );		
	if (! t ) 	return( NULL );
	/* perform ASSIGNMENT on items to be UPDATE [i.e., do the update] */
	t = n->lc->op( n->lc, t );
	/* return  updated items for writing to Typhoon DB */
	return( t );
}

/* SUBSELECT : Left child is a node that returns a TempTable, right child
 * 			is a list of columns to 'keep'. This basically filters
 * 			out the noise of the table_sel :lc is assumed to be a
 * 			table_expr, rc is assumed to be COLS */
struct BSQL_TMPTBL * bsql_subsel_nodeop( struct TREENODE *n, char *data ) {
	struct BSQL_TMPTBL *t, *t_new;
	struct BSQL_COL *col;

	if (! n || ! n->lc || ! n->rc ) 	return(NULL);
	t = n->lc->op(n->lc, data);		/* get temptable from table_expr */
	if (! t ) 	return( NULL );
	col = n->rc->op(n->rc, NULL);		/* get column list */
	if (! col ) 	return( t );

	t_new = bsql_tmptbl_create( col );
	if ( t_new ) {				/* convert table into t_new */
		bsql_tmbtbl_convert( t, t_new );
	}
	bsql_tmptbl_destroy( t );
	bsql_collist_free( col );
	return( t_new );
}

/* TABLESEL : Left child is a node that returns a Temp Table, right child is
 *            a conditional expression applied to each row in table 
 *            Returns a new temp table of expression applied to old table */
/* 	        NOTE: 'data' is assumed to be a TABLE_EXPR_ARG */
struct BSQL_TMPTBL * bsql_tablesel_nodeop( struct TREENODE *n, char *data ) {
	struct BSQL_TMPTBL *t, *t_new;
	struct BSQL_TABLEEXPR_ARG arg;
	struct BSQL_ROW *row;

	if (! n || ! n->lc )
		return(NULL);

	/* get temptable for table */
	t = (struct BSQL_TMPTBL *) n->lc->op( n->lc, data );
	if (! t ) return(NULL);

	if (! n->rc ) 		/* no conditions: return entire table */
		return( t );

	/* build return_val */
	t_new = bsql_tmptbl_create( t->columns );
	if (! t_new ) {
		bsql_tmptbl_destroy( t );
		return(NULL);
	}
	arg.table = t;
	arg.next = NULL;
	row = t->rows;

	while ( row ) {
		arg.row = row->data;
		/* evaluate expression in 'rc' for this row */
		if ( n->rc && n->rc->op( n->rc, &arg ) ) {
			bsql_tmptbl_add( t_new, row->data );
		}
		row = row->next;
	}

	bsql_tmptbl_destroy( t );
	return( t_new );
}

/* JOIN: left child is a 'list' of 2 tables to join, right child is
 *       join-conditional-expression. Returns a new table formed by the
 *       JOIN of the two tables */
/* 	   NOTE: 'data' is not used */
struct BSQL_TMPTBL * bsql_join_nodeop( struct TREENODE *n, void *data ) {
	struct BSQL_TMPTBL *t1, *t2, *t_new;
	struct BSQL_ROW *t1_row, *t2_row;
	struct BSQL_TABLEEXPR_ARG t1arg, t2arg;
	char *row_new;
	int i, j;

	if ( ! n || ! n->lc )
		return(NULL);

	/* check for the existence of two items in the table list */
	if ( ! n->lc->rc || ! n->lc->lc || ! n->lc->lc->rc )
		return(0);
	/* Access op() of TBL_LIST[0] to get first temp-table */
	t1 = (struct BSQL_TMPTBL *) n->lc->op( n->lc, NULL );
	if (! t1 ) return(NULL);

	/* Access op() of TBL_LIST[1] to get first temp-table */
	t2 = (struct BSQL_TMPTBL *) n->lc->lc->op( n->lc->lc, NULL );
	if (! t2 ) { 
		bsql_tmptbl_destroy( t1 ); 
		return(NULL);
	}

	/* construct table args to pass to expr-parser */
	t1arg.table = t1;
	t1arg.next = &t2arg;
	t2arg.table = t2;
	t2arg.next = NULL;
	
	/* build return_val */
	t_new = bsql_tmptbl_create( bsql_tmptbl_colmerge( t1, t2 ) );
	if (! t_new ) {
		bsql_tmptbl_destroy( t1 );
		bsql_tmptbl_destroy( t2 );
		return(NULL);
	}

	t1_row = t1->rows;
	while ( t1_row ) {
		t1arg.row = t1_row->data;
		t2_row = t2->rows;
		while ( t2_row ) {
			t2arg.row = t2_row->data;
			/* evaluate expression for this row */
			if ( n->rc && n->rc->op( n->rc, &t1arg ) ) {
				row_new = bsql_row_merge( t1, t1_row, t2, t2_row ); 
				if ( row_new ) 
					bsql_tmptbl_add( t_new, row_new );
			}
			t2_row = t2_row->next;
		}
		t1_row = t1_row->next;
	}

	bsql_tmptbl_destroy( t1 );
	bsql_tmptbl_destroy( t2 );
	return( t_new );
}



/* ====================================================== TERTIARY NODES */

/* TABLE_CPY: returns a TempTable created from a Typhoon table
 *            The right child is a VAL_REF contining the table;
 *            the left will be selection criteria one day :P */
/* 	        NOTE: 'data' is not used */
struct BSQL_TMPTBL * bsql_tablecpy_nodeop( struct TREENODE *n, void *data ) {
	struct BSQL_TMPTBL *t_new;
	struct BSQL_VALREF *val;
	struct BSQL_COL *cols;
	char *rec;
	int cont;

	if ( !n || n->rc ) 
		return( NULL );
	/* get VALREF containing table info */
	v = n->rc->op( n->rc, NULL ); 
	if ( v->type != TYPE_DBTABLE ) {
		free(v);
		return(NULL);
	}

	/* build return_val */
	cols = bsql_cols_from_dbtable( t->columns );
	t_new = bsql_tmptbl_create( cols );
	bsql_collist_free( cols );
	if (! t_new ) 	return(NULL);

	/* create temp table from DB */
	row = calloc( v->u.db->table->size, 1 );
	if (! row ) {
		bsql_tmptbl_destroy( t_new );
		return(NULL);
	}

	cont = db_index_first( v->u.db_table->index->id, row );
	while ( cont ) {
		bsql_tmptbl_add( t_new, row );
		cont = db_index_next( v->u.db_table->index->id, row );
	}
	free(row);
	free(v);
	return( t_new );
};

/* TBL_LIST NODE : used for listing the columns in an assign statement
 * 			left child = TBL_LIST or NULL
 * 			right child = table_expr returning TMPTBL
 *			RETURNS linked list of columns, rc appended to lc 
 *			NOTE: data is not used*/
struct BSQL_TMPTBL * bsql_tbllst_nodeop( struct TREENODE *n, void *data ) {
	/* this should return a list like COL_LIST, but since table lists
	 * are never used except in JOINS, and since there they only contain
	 * 2 items, JOIN has been written to access these items directly. As
	 * such, each TBL_LIST node will only return the value of its right
	 * child */
	if ( n->rc ) 
		return( n->rc->op( n->rc ) );
	return(NULL);
}

/* COL_LIST NODE : used for listing the columns in an assign statement
 * 			left child = COL_LIST or NULL
 * 			right child = VAL_REF returning COL_REF
 *			RETURNS linked list of columns, rc appended to lc 
 *			NOTE: data is not used*/
struct BSQL_COL * bsql_collst_nodeop( struct TREENODE *n, void *data ) {
	struct BSQL_COL *col, *c;

	if (! n || ! n->rc )		return(NULL);
	/* this is the first node in the list ? */
	if (! n->lc ) return( n->rc->op( n->lc, NULL ) );

	col = n->lc->op( n->rc, NULL );
	/* go to end of col_list */
	c = col;
	while ( c->next ) {
		c = c->next;
	}
	/* add ours to end of list */
	c->next = n->rc->op( n->lc, NULL );
	/* update offset etc */
	if ( c->next )
		c->next->offset = c->offset + (c->size * db_type_sizeof(c->type));
	return( col );
}

/* VALLIST NODE : used for listing values in an assign statement 
 * 			left child = VAL_LIST or NULL
 * 			right child = VAL_REF
 * 			RETURNS linked list of VALREFs, rc appended to lc
 *			data is a TABLE_EXPR_ARG for resolving colref VALREFs */
struct BSQL_VALLIST * bsql_vallst_nodeop( struct TREENODE *n, char *data ) {
	struct BSQL_VALLIST *val, *v;

	if (! n || ! n->rc )		return(NULL);

	/* this is the first node in the list ? */
	if (! n->lc ) return( n->rc->op( n->lc, data ) );

	val = n->lc->op( n->rc, data );
	/* go to end of col_list */
	v = val;
	while ( v->next ) {
		v = v->next;
	}
	/* add ours to end of list */
	v->next = n->rc->op( n->lc, data );
	return( val );
}

/* ASGN
 * 			table is incoming table [target of assignments]
 * 			if NULL, table form lc is returned unmodified
 * 			left child returns TMP_TBL == table_expr or static_row
 * 			right child is col_list
 * 			data is not used?? */
struct BSQL_TMPTBL * bsql_asgn_nodeop( struct TREENODE *n, 
							struct BSQL_TMPTBL *table ) {
	struct BSQL_TMPTBL *t;
	struct BSQL_COL *col_list;
	struct BSQL_ROW *dest_row;
	struct BSQL_TABLEEXPR_ARG arg = {0};
	char *row_data;

	if ( ! n || ! n->lc || ! n->rc ) 	return( 0 );
	col_list = n->rc->op( nc->rc, NULL );
	if (! table ) 
		table = bsql_tmptbl_new(col_list);
	
	dest_row = table->rows;
	arg.table = table;				/* pass row of table to rc */
	while ( dest_row ) {
		/* uh, make arg here ?*/
		arg.row = dest_row->data;
		t = n->lc->op( nc->lc, &arg);		/* generate values each time */
		if ( t ) {
	 		row_data = t->rows->data;	/* assigns only have one row */
			/* assign field from row_data to dest_row */
			bsql_row_assign( row_data, col_list, dest_row->data, 
						table->columns );
			bsql_tmptbl_destroy( t );
		}
		dest_row = dest_row->next;
	}
	bsql_collist_free( col_list );
	return(table);
}

/* STATIC_ROW : returns a single-line table containing a list of values (for
 * 		    assignment). left child is VAL_LIST, right child is NULL 
 * 		    'data' is a TABLE_EXPR_ARG for doing colrefs */
struct BSQL_TMPTBL * bsql_staticrow_nodeop( struct TREENODE *n, char *data ) {
	struct BSQL_TMPTBL *t;
	struct BSQL_COL col_list, *c;
	struct BSQL_VAL_LIST *val_list, *v;
	char *row, *ptr;

	if (! n || ! n->lc )	return(NULL);
	val_list = n->lc->op( n->lc, data );

	/* foreach val, create a col */
	v = val_list;
	c = &col_list;
	while ( v ) {
		c->next = calloc( sizeof(struct BSQL_COL), 1 );
		if (! c) break;
		c = c->next;
		c->type = v->type;
		c->size = v->size;
		c->name = "tmp";
		v = v->next;
	}
	 /* create temp table */
	t = bsql_tmptbl_new( col_list->next );
	bsql_collist_free( col_list->next );
	if (! t ) {
		bsql_vallist_free( val_list );
		return(NULL);
	}

	row = calloc( t->row_size, 1);
	if ( ! row )
		bsql_vallist_free( val_list );
		return(NULL);
	}

	/* foreach value, add to row */
	v = val_list;
	c = t->columns;
	ptr = row;
	while ( v && c ){
		db_set_field_in_rec( c, row, &v->u.ul );
		v = v->next;
		c = c->next;
	}
	bsql_vallist_free( val_list );
		
	free( row );
	return( t );
}



/* ==================================================== BOOLEAN NODES */

int bsql_booland_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	int v1, v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = (int) n->lc->op( n->lc, data );
	v2 = (int) n->rc->op( n->rc, data );
	
	return( v1 && v2 );
}

int bsql_boolor_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	int v1, v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = (int) n->lc->op( n->lc, data );
	v2 = (int) n->rc->op( n->rc, data );
	//if ( v1 || v2 ) return( 1 );
	return( v1 || v2 );
}

int bsql_boolnot_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	int v1;

	if ( ! n->rc )		/* left child is empty */
		return(NULL);
	v1 = (int) n->rc->op( n->rc, data );
	return( ! v1 );
}



/* ==================================================== COMPARISON NODES */
/* generic compare: returns v1->value - v2->value */
int bsql_cmpgen_nodeop( struct BSQL_VALREF *v1, struct BSQL_VALREF *v2 ) {
	long l1, l2;
	unsigned long u1, u2;
	int rv;

	/* for strings, compare them */
	if ( v1->type == v2->type && v1->type == TYPE_CHAR && 
		v1->size > 1 && v2->size > 1 ) {
		return (strcmp( v1->u.str, v2->u.str ) ); 
	} 

	/* other wise convert to signed/unsigned long and compare */
	if ( DB_TYPE_ISSIGNED( v1->type ) ) {
		l1 = db_valtol( &v1->u.ulong, v1->type );
		if ( DB_TYPE_ISSIGNED( v2->type ) ) {
			l2 = db_valtol( &v2->u.ulong, v2->type );
			rv = (int) (l1 - l2);
		} else {		
			u2 = db_valtoul( &v2->u.ulong, v2->type );
			rv = (int) (l1 - (long) u2);
		}
	} else {
		u1 = db_valtoul( &v1->u.ulong, v1->type );
		if ( DB_TYPE_ISSIGNED( v2->type ) ) {
			l2 = db_valtol( &v2->u.ulong, v2->type );
			rv = (int) (u1 - (unsigned long) l2);
		} else {
			u2 = db_valtoul( &v2->u.ulong, v2->type );
			rv = (int) (u1 - u2);
		}
	}
	return(rv);
}

int bsql_cmplt_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *v1, *v2;
	int rv;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_cmpgen_nodeop( v1, v2 );
	free( v1 );
	free( v2 );
	if ( rv < 0 ) return(1);
	return( 0 );
}

int bsql_cmple_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *v1, *v2;
	int rv;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_cmpgen_nodeop( v1, v2 );
	free( v1 );
	free( v2 );
	if ( rv <= 0 ) return(1);
	return( 0 );
}

int bsql_cmpeq_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *v1, *v2;
	int rv;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_cmpgen_nodeop( v1, v2 );
	free( v1 );
	free( v2 );
	if ( rv == 0 ) return(1);
	return( 0 );
}

int bsql_cmpge_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *v1, *v2;
	int rv;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_cmpgen_nodeop( v1, v2 );
	free( v1 );
	free( v2 );
	if ( rv >= 0 ) return(1);
	return( 0 );
}

int bsql_cmpgt_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *v1, *v2;
	int rv;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_cmpgen_nodeop( v1, v2 );
	free( v1 );
	free( v2 );
	if ( rv > 0 ) return(1);
	return( 0 );
}

int bsql_cmpne_nodeop( struct TREENODE *n, struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *v1, *v2;
	int rv;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_cmpgen_nodeop( v1, v2 );
	free( v1 );
	free( v2 );
	if ( rv != 0 ) return(1);
	return( 0 );
}

/* TODO: LIKE, IN comparison */



/* ==================================================== MATHOP NODES */

/* generic math_op functions */
typedef unsigned long (*BSQL_MATHOP_FN)(unsigned long, unsigned long);

unsigned long bsql_mathop_add( unsigned long l1, unsigned long l2 ) {
	return( l1 + l2 );
}

unsigned long bsql_mathop_sub( unsigned long l1, unsigned long l2 ) {
	return( l1 - l2 );
}

unsigned long bsql_mathop_mul( unsigned long l1, unsigned long l2 ) {
	return( l1 * l2 );
}

unsigned long bsql_mathop_div( unsigned long l1, unsigned long l2 ) {
	return( l1 / l2 );
}

unsigned long bsql_mathop_and( unsigned long l1, unsigned long l2 ) {
	return( l1 & l2 );
}

unsigned long bsql_mathop_or( unsigned long l1, unsigned long l2 ) {
	return( l1 | l2 );
}

unsigned long bsql_mathop_xor( unsigned long l1, unsigned long l2 ) {
	return( l1 ^ l2 );
}


/* generic math function: converts to longs and calls math_op function */
struct BSQL_VALREF * bsql_mathgen_nodeop( struct BSQL_VALREF *v1, 
					struct BSQL_VALREF *v2, BSQL_MATHOP_FN fn ) {
	unsigned long l1, l2;
	struct BSQL_VALREF *val;

	/* for strings and arrays, return error */
	if (  v1->size > 1 || v2->size > 1 ) {
		return ( NULL ); 
	} 

	/* otherwise convert to signed/unsigned long and perform op */
	val = calloc( sizeof(struct BSQL_VALREF), 1 );
	l1 = db_valtoul( v1->value, v1->type );
	l2 = db_valtoul( v2->value, v2->type );
	
	val->u.ul = (*fn)( l1, l2 );
	val->type = TYPE_ULONG;
	val->size = 1;
	return( val );
}



struct BSQL_VALREF * bsql_mathadd_nodeop( struct TREENODE *n, 
							 struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *rv, *v1, *v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_mathgen_nodeop( v1, v2, bsql_mathop_add ); 
	free( v1 );
	free( v2 );
	return( rv );
}

struct BSQL_VALREF * bsql_mathsub_nodeop( struct TREENODE *n, 
							 struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *rv, *v1, *v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_mathgen_nodeop( v1, v2, bsql_mathop_sub ); 
	free( v1 );
	free( v2 );
	return( rv );
}

struct BSQL_VALREF * bsql_mathdiv_nodeop( struct TREENODE *n, 
							 struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *rv, *v1, *v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_mathgen_nodeop( v1, v2, bsql_mathop_div ); 
	free( v1 );
	free( v2 );
	return( rv );
}

struct BSQL_VALREF * bsql_mathmul_nodeop( struct TREENODE *n, 
							 struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *rv, *v1, *v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_mathgen_nodeop( v1, v2, bsql_mathop_mul ); 
	free( v1 );
	free( v2 );
	return( rv );
}

struct BSQL_VALREF * bsql_mathand_nodeop( struct TREENODE *n, 
							 struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *rv, *v1, *v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_mathgen_nodeop( v1, v2, bsql_mathop_and ); 
	free( v1 );
	free( v2 );
	return( rv );
}

struct BSQL_VALREF * bsql_mathor_nodeop( struct TREENODE *n, 
							 struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *rv, *v1, *v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_mathgen_nodeop( v1, v2, bsql_mathop_or ); 
	free( v1 );
	free( v2 );
	return( rv );
}

struct BSQL_VALREF * bsql_mathxor_nodeop( struct TREENODE *n, 
							 struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *rv, *v1, *v2;

	if ( ! n->lc  || ! n->rc )
		return(NULL);
	v1 = n->lc->op( n->lc, data );
	v2 = n->rc->op( n->rc, data );
	rv = bsql_mathgen_nodeop( v1, v2, bsql_mathop_xor ); 
	free( v1 );
	free( v2 );
	return( rv );
}




/* =================================================== LEAF NODE */
struct BSQL_VALREF * bsql_valref_nodeop( struct TREENODE *n, 
							 struct BSQL_TABLEEXPR_ARG *data ) {
	struct BSQL_VALREF *rv;
	struct BSQL_COL *col;
	struct TREELEAF *leaf = (struct TREELEAF *)n; 

	rv = calloc( sizeof(struct BSQL_VALREF), 1 );
	if (! rv) 	return(NULL);
	rv->type = leaf->type;
	rv->size = leaf->size;
	switch ( leaf->type ) {
		case TYPE_COLREF:
			if ( data ) {
				/* get value from current record in arg */
				rv = bsql_val_getfromarg( leaf->u.col, data );
			} else {
				/* just return COL structure */
				free(rv);
				col = calloc( sizeof( struct BSQL_COL ), 1 );
				if ( col && leaf->u.col ) {
					memcpy(col, leaf->u.col,sizeof(struct BSQL_COL));
				}
				col->next = NULL;
				return( (struct BSQL_VALREF *) col );
			break;
		case TYPE_DBTABLE:
			rv->u.db_table = leaf->u.db_table;
			rv->size = 1;
			break;
		case TYPE_TABLEREF:
			rv->u.table = leaf->u.table;
			rv->size = 1;
			break;
		case TYPE_STRINGREF: 
			rv->u.str = leaf->u.str;
			rv->size = strlen( leaf->u.str );
			rv->type = TYPE_CHAR;
			break;
		case TYPE_CHAR: 
			rv->u.c = leaf->u.c;
			break;
		case TYPE_UCHAR: 
			rv->u.uc = leaf->u.uc; break;
		case TYPE_SHORT: 
			rv->u.s = leaf->u.s; break;
		case TYPE_USHORT: 
			rv->u.us = leaf->u.us; break;
		case TYPE_LONG: 
			rv->u.l = leaf->u.l; break;
		case TYPE_ULONG: 
			rv->u.ul = leaf->u.ul; break;
		case TYPE_UINT: 
			rv->u.ui = leaf->u.ui; break;
		case TYPE_INT: 
		default:
			/* no need to lookup, just return pointer to leaf value */
			rv = (BSQL_VALREF *) calloc( sizeof(struct BSQL_VALREF), 1);
			rv->type = TYPE_INT;
			rv->size = 1;
			rv->u.i = &leaf->u.i;
			break;
	}
	return( rv );
}




/* ===================================================== TREE NODE Creation */

void bsql_newnode_setchildren( struct TREENODE *n, void *l, void *r ) {
	n->lc = (struct TREENODE *)l;
	n->rc = (struct TREENODE *)r;
	if ( l ) (struct TREENODE *)l->p = n;
	if ( r ) (struct TREENODE *)r->p = n;
	return;
}

/* primary nodes */
struct TREENODE * bsql_new_insert( struct TREENODE *asgn, 
							struct TREELEAF *db ) {
	struct TREENODE *new;
	if (! left || ! db ) return(NULL);
	if ( asgn->op != bsql_insertasgn_nodeop  ||
		db->type != TYPE_DBTABLE)
			return(NULL);
	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);
	new->op = bsql_insert_nodeop;
	bsql_node_setchildren( new, asgn, db );
	return(new);
}

struct TREENODE * bsql_new_update( struct TREENODE *asgn, 
							struct TREELEAF *db ) {
	struct TREENODE *new;
	if (! asgn || ! db ) return(NULL);
	if ( asgn->op != bsql_updateasgn_nodeop ||
		db->type != TYPE_DBTABLE )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);
	new->op = bsql_updateasgn_nodeop;
	bsql_node_setchildren( new, asgn, db );
	return(new);
}

struct TREENODE * bsql_new_delete( struct TREENODE *items, 
							struct TREELEAF *db ) {
	struct TREENODE *new;
	if (! items || ! db ) return(NULL);
	if ( ( items->op != bsql_tablesel_nodeop && 
	     items->op != bsql_subsel_nodeop  ) || db->type != TYPE_DBTABLE )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_delete_nodeop;
	bsql_node_setchildren( new, items, db );
	return(new);
}

struct TREENODE * bsql_new_select( struct TREENODE *tblexpr ) {
	struct TREENODE *new;
	if (! tblexpr ) return(NULL);
	if ( tblexpr->op != bsql_tablesel_nodeop && 
		tblexpr->op !=bsql_subsel_nodeop )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_select_nodeop;
	bsql_node_setchildren( new, tblexpr, NULL );
	return(new);
}

/* secondary nodes */
struct TREENODE * bsql_new_insertasgn( struct TREENODE *expr, 
							struct TREENODE *asgn ) {
	struct TREENODE *new;
	if (! expr || ! asgn ) return(NULL);
	if ( expr->op != bsql_subsel_nodeop || asgn->op != bsql_asgn_nodeop)
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_insertasgn_nodeop;
	bsql_node_setchildren( new, expr, asgn );
	return(new);
}

struct TREENODE * bsql_new_updateasgn( struct TREENODE *expr, 
							struct TREENODE *asgn ) {
	struct TREENODE *new;
	if (! expr || ! asgn ) return(NULL);
	if ( expr->op != bsql_subsel_nodeop || asgn->op != bsql_asgn_nodeop)
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_updateasgn_nodeop;
	bsql_node_setchildren( new, expr, asgn );
	return(new);
}

struct TREENODE * bsql_new_subsel( struct TREENODE *tblexpr, 
							struct TREENODE *collst ) {
	struct TREENODE *new;
	if (! tblexpr || ! collst ) return(NULL);
	if ( tblexpr->op != bsql_tablesel_nodeop ||
		collst->op != bsql_collst_nodeop )
		return(NULL);
	
	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_subsel_nodeop;
	bsql_node_setchildren( new, tblexpr, collst );
	return(new);
}
	
struct TREENODE * bsql_new_tablesel( struct TREENODE *tmptbl, 
							struct TREENODE *cond ) {
	struct TREENODE *new;
	if (! tmptbl || ! cond ) return(NULL);
	if ( ! IS_TMPTBL_NODE( tmptbl )  || ! IS_BOOL_NODE( cond ) || 
		! IS_COND_NODE( cond ) )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_tablesel_nodeop;
	bsql_node_setchildren( new, tmptbl, cond );
	return( new );
}

struct TREENODE * bsql_new_join( struct TREENODE *t1, struct TREENODE *t2, 
							struct TREENODE *cond ) {
	struct TREENODE *new, *tmp;
	if (! t1 || !t2 || ! cond ) return(NULL);
	if (! IS_TMPTBL( t1 ) || ! IS_TMPTBL( t2 ) || ! IS_COND_NODE( cond ) )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_join_nodeop;
	/* make 2-element table list for t1 and t2 */
	tmp = bsql_new_tbllst( NULL, t2 );
	new->lc = bsql_new_tbllst( tmp, t1 );
	
	bsql_node_setchildren( new, new->lc, cond );
	return( new );
}

/* tertiary nodes */

struct TREENODE * bsql_new_tablecpy( struct TREELEAF *table ) {
	struct TREENODE *new;
	if ( ! table ) return(NULL);
	if ( table->type  != TYPE_DBTABLE )
		return(NULL);
	
	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_tablecpy_nodeop;
	bsql_node_setchildren( new, NULL, table );
	return( new );
}

struct TREENODE * bsql_new_tbllst( struct TREENODE *list, 
							struct TREENODE *table ) {
	struct TREENODE *new;
	if (! table ) return(NULL);
	if ( ( list && list->op != bsql_tbllst_nodeop ) ||
		! IS_TMPTBL_NODE( table )  )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_tbllst_nodeop;
	bsql_node_setchildren( new, list, table );
	return(new);
}

struct TREENODE * bsql_new_collst( struct TREENODE *list, 
							struct TREELEAF *col ) {
	struct TREENODE *new;
	if (! col ) return(NULL);
	if ( ( list && list->op != bsql_collst_nodeop ) ||
		col->type != TYPE_COLREF )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_collst_nodeop;
	bsql_node_setchildren( new, list, col );
	return(new);
}

struct TREENODE * bsql_new_vallst( struct TREENODE *list, 
							struct TREELEAF *val ) {
	struct TREENODE *new;
	if (! val ) return(NULL);
	if ( list && list->op != bsql_vallst_nodeop ) 
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_vallst_nodeop;
	bsql_node_setchildren( new, list, val );
	return(new);
}

struct TREENODE * bsql_new_asgn( struct TREENODE *asgn, 
							struct TREENODE *cols ) {
	struct TREENODE *new;
	if (! asgn || ! cols ) return(NULL);
	if ( (! IS_TMPTBL( asgn ) && asgn->op != bsql_staticrow_nodeop) ||
		cols->op != bsql_collst_nodeop)
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_asgn_nodeop;
	bsql_node_setchildren( new, asgn, cols );
	return(new);
}

struct TREENODE * bsql_new_staticrow( struct TREENODE *vals ) { 
	struct TREENODE *new;
	if (! vals ) return(NULL);
	if ( vals->op != bsql_vallst_nodeop ) 	return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	new->op = bsql_staticrow_nodeop;
	bsql_node_setchildren( new, vals, NULL );
	return(new);
}

/* boolean nodes */
struct TREENODE * bsql_new_bool( struct TREENODE *left, 
					struct TREENODE *right, int type ) {
	struct TREENODE *new;

	if ( ! right || (! IS_BOOL_NODE(right) && ! IS_COND_NODE(right)) )
		return(NULL);
	if ( (! left && type != BSQL_BOOL_NOT ) || ( left && 
			(! IS_BOOL_NODE(left) && ! IS_COND_NODE(left)) ) )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	switch ( type ) {
		case BSQL_BOOL_AND:
			new->op = bsql_booland_nodeop; break;
		case BSQL_BOOL_OR:
			new->op = bsql_boolor_nodeop; break;
		case BSQL_BOOL_NOT:
			new->op = bsql_boolnot_nodeop; break;
	}
	bsql_node_setchildren( new, left, right );
	return(new);
}

/* comparison nodes */
struct TREENODE * bsql_new_cmp( struct TREENODE *left, 
					struct TREENODE *right, int type ) {
	struct TREENODE *new;
	if (! left || ! right ) return(NULL);
	if ( (! IS_MATHOP_NODE(left) && left->op != bsql_valref_nodeop) ||
		(! IS_MATHOP_NODE(right) && right->op != bsql_valref_nodeop) )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	switch ( type ) {
		case BSQL_CMP_LT:
			new->op = bsql_cmplt_nodeop; break;
		case BSQL_CMP_LE:
			new->op = bsql_cmple_nodeop; break;
		case BSQL_CMP_EQ:
			new->op = bsql_cmpeq_nodeop; break;
		case BSQL_CMP_GE:
			new->op = bsql_cmpge_nodeop; break;
		case BSQL_CMP_GT:
			new->op = bsql_cmpgt_nodeop; break;
		case BSQL_CMP_NE:
			new->op = bsql_cmpne_nodeop; break;
	}
	bsql_node_setchildren( new, left, right );
	return(new);
}

/* mathop nodes */
struct TREENODE * bsql_new_mathop( struct TREENODE *left, 
					struct TREENODE *right, int type ) {
	struct TREENODE *new;
	if (! left || ! right ) return(NULL);
	if ( (! IS_MATHOP_NODE(left) && left->op != bsql_valref_nodeop) ||
		(! IS_MATHOP_NODE(right) && right->op != bsql_valref_nodeop) )
		return(NULL);

	new = calloc( sizeof(struct TREENODE), 1 );
	if (! new) return(NULL);

	switch ( type ) {
		case BSQL_MATH_ADD:
			new->op = bsql_mathadd_nodeop; break;
		case BSQL_MATH_SUB:
			new->op = bsql_mathsub_nodeop; break;
		case BSQL_MATH_MUL:
			new->op = bsql_mathmul_nodeop; break;
		case BSQL_MATH_DIV:
			new->op = bsql_mathdiv_nodeop; break;
		case BSQL_MATH_AND:
			new->op = bsql_mathand_nodeop; break;
		case BSQL_MATH_OR:
			new->op = bsql_mathor_nodeop; break;
		case BSQL_MATH_XOR:
			new->op = bsql_mathxor_nodeop; break;
	}
	bsql_node_setchildren( new, left, right );
	return(new);
}

/* =========================================== New LeafNodes */
struct TREELEAF * bsql_new_dbtable( int id ) {
	struct TREELEAF *leaf = calloc(sizeof(struct TREELEAF), 1); 
	if (! leaf ) return(NULL);
	leaf->type = TYPE_DBTABLE;
	leaf->op = bsql_valref_nodeop;
	leaf->u.db_table = id;
	return( leaf );
}

struct TREELEAF * bsql_new_colref( struct BSQL_COL *col ) {
	struct TREELEAF *leaf = calloc(sizeof(struct TREELEAF), 1); 
	if (! leaf ) return(NULL);
	leaf->type = TYPE_COLREF;
	leaf->op = bsql_valref_nodeop;
	leaf->u.col = col;
	return( leaf );
}

struct TREELEAF * bsql_new_tableref( struct BSQL_TMPTBL *t ) {
	struct TREELEAF *leaf = calloc(sizeof(struct TREELEAF), 1); 
	if (! leaf ) return(NULL);
	leaf->type = TYPE_TABLEREF;
	leaf->op = bsql_valref_nodeop;
	leaf->u.table = t;
	return( leaf );
}

struct TREELEAF * bsql_new_string( char *str ) {
	struct TREELEAF *leaf = calloc(sizeof(struct TREELEAF), 1); 
	if (! leaf ) return(NULL);
	leaf->type = TYPE_STRINGREF;
	leaf->op = bsql_valref_nodeop;
	leaf->u.str = str;
	return( leaf );
}

struct TREELEAF * bsql_new_lvalue( unsigned long val ) {
	struct TREELEAF *leaf = calloc(sizeof(struct TREELEAF), 1); 
	if (! leaf ) return(NULL);
	leaf->type = TYPE_ULONG;
	leaf->op = bsql_valref_nodeop;
	leaf->u.db_table = val;
	return( leaf );
}
