#include <string.h>
#include <unistd.h>
#include <stdlib.h>



/* -------------------------------------------------------- elf structs  */
struct ELF_HDR {
	char e_ident[16];	/* Magic number and other info */
	short e_type;		/* Object file type */
	short e_machine;	/* Architecture */
	int e_version;		/* Object file version */
	int e_entry;		/* Entry point virtual address */
	int e_phoff;		/* Program header table file offset */
	int e_shoff;		/* Section header table file offset */
	int e_flags;		/* Processor-specific flags */
	short e_ehsize;		/* ELF header size in bytes */
	short e_phentsize;	/* Program header table entry size */
	short e_phnum;		/* Program header table entry count */
	short e_shentsize;	/* Section header table entry size */
	short e_shnum;		/* Section header table entry count */
	short e_shstrndx;	/* Section header string table index */
};

struct ELF_SECT {
	int sh_name;		/* Section name (string tbl index) */
	int sh_type;		/* Section type */
	int sh_flags;		/* Section flags */
	int sh_addr;		/* Section virtual addr at execution */
	int sh_offset;		/* Section file offset */
	int sh_size;		/* Section size in bytes */
	int sh_link;		/* Link to another section */
	int sh_info;		/* Additional section information */
	int sh_addralign;	/* Section alignment */
	int sh_entsize;		/* Entry size if section holds table */
};


struct ELF_PROG {
	int p_type;		/* Segment type */
	int p_offset;		/* Segment file offset */
	int p_vaddr;		/* Segment virtual address */
	int p_paddr;		/* Segment physical address */
	int p_filesz;		/* Segment size in file */
	int p_memsz;		/* Segment size in memory */
	int p_flags;		/* Segment flags */
	int p_align;		/* Segment alignment */
};

struct ELF_IDENT {
	char ei_magic[4];
	char ei_class;
	char ei_data;
	char ei_version;
};

struct ELF_DYN {
	int tag;
	int val;
};

struct DYNSTR {
	int offset;
	struct DYNSTR *next;
};

struct ELF_SYM {
	int st_name;
	int st_value;
	int st_size;
	char st_info;
	char st_other;
	short st_shndx;
};

/* --------------------------------------------------- passed to lib */
struct ELF_DYNAMIC_INFO {
	char *soname;
	char *rpath;
	/* special sections */
	int debug, init, fini;
	int sym_addr, sym_size;
	int str_addr, str_size;
	int hash_addr;
	/* relocation information */
	int rela_addr, rela_size, rela_ent;
	int rel_addr, rel_size, rel_ent;
	int text_rel;
	/* PLT info */
	int plt_size, plt_addr, plt_type, plt_rel;
};

struct ELF_STRING_BUF {
	char *elf_str;
	char *sht_str;
	char *pht_str;
	char *note;
	int e_len, s_len, p_len, n_len;
};

typedef int (*new_addr_fn) (long, int, long, int);
typedef int (*new_name_fn) (long, char *, int);
typedef int (*new_imp_fn) (long, char *, char *, int);
typedef int (*new_lib_fn) (char *, int, int);
typedef int (*new_fun_fn) (long, char *, int, int);
typedef int (*new_sec_fn) (char *, long, int, long, int);
typedef int (*struct_fn) (long, unsigned long);

struct ELF_INFO {
	int fd;			/* file descriptor of target */
	long base;		/* base of ELF file in fd */
	/* these are fn pointers instead of calls to bastard API */
	new_addr_fn addr_new;
	new_name_fn name_new;
	new_imp_fn imp_new;
	new_lib_fn lib_new;
	new_fun_fn func_new;
	new_sec_fn sec_new;
	struct_fn  struct_apply;
	/* this is a collection of string buffers */
	struct ELF_STRING_BUF str;
	/* these are return values from parse_elf() */
	struct ELF_HDR header;
	struct ELF_DYNAMIC_INFO dynamic;
} *opt;

/* ---------------------------------------------------------- symbol types */
#define STT_NOTYPE       0
#define STT_OBJECT       1
#define STT_FUNC         2
#define STT_SECTION      3
#define STT_FILE         4
#define STT_LOPROC      13
#define STT_HIPROC      15


/* --------------------------------------------------------------- errors */
#define EBADPARAM  -1
#define EBADELF    -2

/* --------------------------------------------------------------- globals */
char strMagic[4] = { 0x7F, 'E', 'L', 'F' };
char *e_strtable = NULL;
int e_strtab_size;
char buf[512], fbuf[8];		/* temporary storage */

/* ----------------------------------------- Constant-to-String Conversions */
const char *get_sh_type(int sh_type)
{
	/*   Convert Type to printable string    */
	switch (sh_type) {
	case 0:
		return ("SHT_NULL");
	case 1:
		return ("SHT_PROGBITS");
	case 2:
		return ("SHT_SYMTAB");
	case 3:
		return ("SHT_STRTAB");
	case 4:
		return ("SHT_RELA");
	case 5:
		return ("SHT_HASH");
	case 6:
		return ("SHT_DYNAMIC");
	case 7:
		return ("SHT_NOTE");
	case 8:
		return ("SHT_NOBITS");
	case 9:
		return ("SHT_REL");
	case 10:
		return ("SHT_SHLIB");
	case 11:
		return ("SHT_DYNSYM");
	case 12:
		return ("SHT_NUM");
	case 0x60000000:
		return ("SHT_LOOS");
	case 0x6fffffff:
		return ("SHT_HIOS");
	case 0x70000000:
		return ("SHT_LOPROC");
	case 0x7fffffff:
		return ("SHT_HIPROC");
	case 0x80000000:
		return ("SHT_LOUSER");
	case 0x8fffffff:
		return ("SHT_HIUSER");
	default:
		return ("unknown");
	}
}

const char *get_sh_flags(int flags)
{
	/*   Convert Flags to printable string    */
	sprintf(fbuf, "---");
	if (flags & 1)
		fbuf[0] = 'w';
	if (flags & 2)
		fbuf[1] = 'a';
	if (flags & 4)
		fbuf[2] = 'x';
	return (fbuf);
}

const char *get_ph_type(int ph_type)
{
	switch (ph_type) {
	case 0:
		return ("PT_NULL");
	case 1:
		return ("PT_LOAD");
	case 2:
		return ("PT_DYNAMIC");
	case 3:
		return ("PT_INTERP");
	case 4:
		return ("PT_NOTE");
	case 5:
		return ("PT_SHLIB");
	case 6:
		return ("PT_PHDR");
	case 7:
		return ("PT_NUM ");
	case 0x60000000:
		return ("PT_LOOS");
	case 0x6FFFFFFF:
		return ("PT_HIOS");
	case 0x70000000:
		return ("PT_LOWPROC");
	case 0x7FFFFFFF:
		return ("PT_HIPROC");
	default:
		return ("unknown");
	}
}

const char *get_ph_flags(int flags)
{
	/*   Convert Flags to printable string    */
	sprintf(fbuf, "---");
	if (flags & 1)
		fbuf[0] = 'x';
	if (flags & 2)
		fbuf[1] = 'w';
	if (flags & 4)
		fbuf[2] = 'r';
	return (fbuf);
}

const char *get_mach_name(short e_machine)
{
	/* this has to be fixed to not be so lame */
	switch (e_machine) {
	case 0:
		return ("EM_NONE");
	case 1:
		return ("EM_M32");
	case 2:
		return ("EM_SPARC");
	case 3:
		return ("EM_386");
	case 4:
		return ("EM_68K");
	case 5:
		return ("EM_88K");
	case 6:
		return ("EM_486");
	case 7:
		return ("EM_860");
	case 8:
		return ("EM_MIPS");
	case 9:
		return ("EM_S370");
	case 10:
		return ("EM_MIPS_RS4_BE");
	case 11:
		return ("EM_RS6000");
	case 15:
		return ("EM_PARISC");
	case 16:
		return ("EM_nCUBE");
	case 17:
		return ("EM_VPP500");
	case 18:
		return ("EM_SPARC32PLUS");
	case 19:
		return ("EM_960");
	case 20:
		return ("EM_PPC");
	case 36:
		return ("EM_V800");
	case 37:
		return ("EM_FR20");
	case 38:
		return ("EM_RH32");
	case 39:
		return ("EM_MMA");
	case 40:
		return ("EM_ARM");
	case 41:
		return ("EM_FAKE_ALPHA");
	case 42:
		return ("EM_SH");
	case 43:
		return ("EM_SPARCV9");
	case 44:
		return ("EM_TRICORE");
	case 45:
		return ("EM_ARC");
	case 46:
		return ("EM_H8_300");
	case 47:
		return ("EM_H8_300H");
	case 48:
		return ("EM_H8S");
	case 49:
		return ("EM_H8_500");
	case 50:
		return ("EM_IA_64");
	case 51:
		return ("EM_MIPS_X");
	case 52:
		return ("EM_COLDFIRE");
	case 53:
		return ("EM_68HC12");
	case 54:
		return ("EM_NUM");
	case 0x9026:
		return ("EM_ALPHA");
	default:
		return ("unknown");
	}
}

const char *get_elf_type(short e_type)
{
	switch (e_type) {
	case 0:
		return ("ET_NONE");
	case 1:
		return ("ET_REL");
	case 2:
		return ("ET_EXEC");
	case 3:
		return ("ET_DYN");
	case 4:
		return ("ET_CORE");
	case 5:
		return ("ET_NUM");
	case 0xFF00:
		return ("ET_LOPROC");
	case 0xFFFF:
		return ("ET_HIPROC");
	default:
		return ("unknown");
	}
}


const char *get_elf_class(int ei_class)
{
	switch (ei_class) {
	case 1:
		return ("CLASS32");
	case 2:
		return ("CLASS64");
	default:
		return ("CLASSNONE");
	}
}

const char *get_elf_data(int ei_data)
{
	switch (ei_data) {
	case 1:
		return ("DATA2LSB");
	case 2:
		return ("DATA2MSB");
	default:
		return ("DATANONE");
	}
}

/* ---------------------------------------------------- UHHHHHH */
struct ELF_SECT **g_sht = NULL;	/* internal 'global' copy of sht */
struct ELF_PROG **g_pht = NULL;	/* internal 'global' copy of pht */
struct ELF_SECT **create_sht(struct ELF_HDR *hdr); /* fwd decl */
struct ELF_PROG **create_pht(struct ELF_HDR *hdr); /* fwd decl */

int get_file_offset(int addr, struct ELF_HDR *hdr)
{
	struct ELF_PROG *p;
	struct ELF_SECT *s;
	int x;
	if (! g_pht) g_pht = create_pht(hdr);
	for (x = 0; x < hdr->e_phnum; x++) {
		p = g_pht[x];
		if (addr >= p->p_vaddr && addr < (p->p_vaddr + p->p_filesz)) {
			return ((addr - p->p_vaddr) + p->p_offset);
		}
	}

	/* .a files have no pht; we do those here */
	if (! g_sht) g_sht = create_sht(hdr);
	for (x = 0; g_sht[x] && x < hdr->e_shnum; x++) {
		s = g_sht[x];
		if (addr >= s->sh_addr && addr < (s->sh_addr + s->sh_size)) {
			return ((addr - s->sh_addr) + s->sh_offset);
		}
	}
	return (0);
}

/* --------------------------------------------------------- SYMBOL TABLE */
struct ELF_SYM **create_symt(int offset, int count)
{
	struct ELF_SYM **symt;
	int x;

	symt = calloc(sizeof (void *), count);

	for (x = 0; x < count; x++) {
		symt[x] = malloc(sizeof (struct ELF_SYM));
		lseek(opt->fd, opt->base + offset + (sizeof (struct ELF_SYM) * x), SEEK_SET);
		read(opt->fd, symt[x], sizeof (struct ELF_SYM));
	}
	return (symt);
}

int free_symt(struct ELF_SYM **symt, int count)
{
	int x;
	if (!symt)
		return (0);
	for (x = 0; x < count; x++)
		if (symt[x])
			free(symt[x]);
	free(symt);
	return (1);
}

/*  ------------------------doSymtab( int, int, int  char *, struct DYNSTR * )
   -- Parse .symtab section
*/
int do_symtab(int sym_addr, int sym_size, char *strtable, struct DYNSTR *libs,
	      struct ELF_HDR *hdr)
{
	struct ELF_SYM **symt, *sym;
	int x, count;
	char *c, *tmp;
	struct DYNSTR *pLIB;

	/* ___________________Here_Is_Where_Names_Are_Created_For_Symbols____ */
	if (!strtable || !sym_addr)
		return (0);

	lseek(opt->fd, opt->base + sym_addr, SEEK_SET);
	count = sym_size / sizeof (struct ELF_SYM);
	symt = create_symt(sym_addr, count);

	for (x = 0; x < count; x++) {
		sym = symt[x];

		/* if it is a valid freaken name.... */
		c = strtable + sym->st_name;
		if (sym->st_name > 0 && c[0] > 31 && c[0] < 127) {
			switch ((sym->st_info & 0xf)) {
			case STT_OBJECT:	/* symbol is a DATA OBJECT */

				(*opt->addr_new) (sym->st_value, sym->st_size,
						  get_file_offset(sym->st_value,
								  hdr),
						  ADDR_SYMBOL | ADDR_DATA);
				(*opt->name_new) (sym->st_value,
						  strtable + sym->st_name,
						  NAME_AUTO | NAME_SYMBOL);
				break;
			case STT_FUNC:	/* Symbol is a FUNCTION */
				/* create an address so we have real PA */
				(*opt->addr_new) (sym->st_value, sym->st_size,
						  get_file_offset(sym->st_value, hdr),
						  ADDR_SYMBOL | ADDR_CODE);
				if (! sym->st_shndx ) {
					/* this is an imported function */
					/* check which DT_NEEDED this falls under */
					tmp = NULL;
					if ( libs ) {
						pLIB = libs->next;
						while (pLIB) {
							if (sym->st_name > pLIB->offset) {
								if (!pLIB->next ||
							    	     pLIB->next->offset >
								     sym->st_name) {
									tmp =
								    	strtable + pLIB->offset;
								}
							}
							pLIB = pLIB->next;
						}
					}
					/* tmp == DT_NEEDED == library owning this func */
					(*opt->imp_new) (sym->st_value, tmp,
							     strtable +
							     sym->st_name, 0);
				} else {
					/* Not an imported function */
					(*opt->func_new) (sym->st_value,
							      strtable +
							      sym->st_name,
							      sym->st_size,
							      NAME_AUTO |
							      NAME_SYMBOL);
				}
				break;
			case STT_FILE:	/* Symbol is a SOURCE FILE */
				/* srcfile = strtable + sym->st_name; */
				break;
			case STT_SECTION:
			case STT_NOTYPE:
				/* add extra symbols [those w/o rva's are discarded] */
				if ( sym->st_shndx ) {
					(*opt->addr_new) (sym->st_value, sym->st_size,
						  get_file_offset(sym->st_value, hdr),
						  ADDR_SYMBOL | ADDR_DATA);
					(*opt->name_new) (sym->st_value,
						      strtable + sym->st_name,
						      NAME_AUTO | NAME_SYMBOL);
				} else {
				/* note: those with no shndx are imports */
					(*opt->imp_new) (sym->st_value, NULL,
							     strtable +
							     sym->st_name, 0);
				}
			case STT_LOPROC:
			case STT_HIPROC:
			default:
				break;
			}	//switch
		}		//if
	}			//for

	free_symt(symt, count);
	return (1);
}

/* -------------------------------------------------------- DYNAMIC SECTION */
struct ELF_DYN **create_dynt(struct ELF_PROG *phdr, int count)
{
	struct ELF_DYN **dynt;
	int x;

	dynt = calloc(sizeof (void *), count);

	for (x = 0; x < count; x++) {
		dynt[x] = malloc(sizeof (struct ELF_DYN));
		lseek(opt->fd, opt->base + phdr->p_offset + (sizeof (struct ELF_DYN) * x), SEEK_SET);
		read(opt->fd, dynt[x], sizeof (struct ELF_DYN));
	}
	return (dynt);
}

int free_dynt(struct ELF_DYN **dynt, int count)
{
	int x;
	if (!dynt)
		return (0);
	for (x = 0; x < count; x++)
		if (dynt[x])
			free(dynt[x]);
	free(dynt);
	return (1);
}

/* These are global linked lists used in parsing _DYNAMIC */
struct DYNSTR *pRPATH, *pNEEDED;

int add_dyn_info(struct ELF_DYNAMIC_INFO *info, struct ELF_DYN *dyn)
{
	switch (dyn->tag) {
	case 0:		//DT_NULL : ignore
		break;
	case 1:		//DT_NEEDED: strname of req lib [opt]
		/* create Dynamic String object, add to NEEDED linked list */
		pNEEDED->next = malloc(sizeof (struct DYNSTR));
		pNEEDED->next->offset = dyn->val;
		pNEEDED->next->next = NULL;
		pNEEDED = pNEEDED->next;
		break;
	case 2:		//DT_PLTRELSZ : val == size of relocs in PLT [opt]
		info->plt_size = dyn->val;
		break;
	case 3:		//DT_PLTGOT : val == ptr to PLT/GOT [opt]
		info->plt_addr = dyn->val;
		break;
	case 4:		//DT_HASH : addr of symbol hash table [req]
		info->hash_addr = dyn->val;
		sprintf(buf, "Symbol hash table at %08X\n", dyn->val);
		strncat(opt->str.note, buf, opt->str.n_len);
		break;
	case 5:		//DT_STRTAB : addr of string table [req]
		info->str_addr = dyn->val;
		break;
	case 6:		//DT_SYMTAB : val = addr of symbol table [req]
		info->sym_addr = dyn->val;
		break;
	case 7:		//DT_RELA : val = addr of reloc table [req]
		info->rela_addr = dyn->val;
		break;
	case 8:		//DT_RELASZ : size of RELA reloc table [reg]
		info->rela_size = dyn->val;
		break;
	case 9:		//DT_RELAENT : size of RELA reloc entry [req]
		info->rela_ent = dyn->val;
		break;
	case 10:		//DT_STRSZ : size of str table [req]
		info->str_size = dyn->val;
		break;
	case 11:		//DT_SYMENT : size of sym table entry [req]
		info->sym_size = dyn->val;
		break;
	case 12:		//DT_INIT : val = addr of init function [opt]
		info->init = dyn->val;
		sprintf(buf, ".init routine at %08X\n", dyn->val);
		strncat(opt->str.note, buf, opt->str.n_len);
		(*opt->addr_new) (dyn->val, 1, dyn->val,
				  ADDR_SYMBOL | ADDR_CODE);
		//(*opt->name_new)( dyn->val, "elf_init", 0x23);      
		(*opt->func_new) (dyn->val, "elf_init",
				  0, NAME_AUTO | NAME_SYMBOL);
		break;
	case 13:		//DT_FINI : val = addr of term function [opt]
		info->fini = dyn->val;
		sprintf(buf, ".fini routine at %08X\n", dyn->val);
		strncat(opt->str.note, buf, opt->str.n_len);
		(*opt->addr_new) (dyn->val, 1, dyn->val,
				  ADDR_SYMBOL | ADDR_CODE);
		//(*opt->name_new)( dyn->val, "elf_fini", 0x23);      
		(*opt->func_new) (dyn->val, "elf_fini",
				  0, NAME_AUTO | NAME_SYMBOL);
		break;
	case 14:		//DT_SONAME : strtable offset of file's soname [ig]
		/* create Dynamic String object, add to SONAME linked list */
		info->soname = (char *) dyn->val;
		break;
	case 15:		//DT_RPATH : val = strtable offset of lib path [opt]
		/* create Dynamic String object, add to RPATH linked list */
		pRPATH->next = malloc(sizeof (struct DYNSTR));
		pRPATH->next->offset = dyn->val;
		pRPATH->next->next = NULL;
		pRPATH = pRPATH->next;
		break;
	case 16:		//DT_SYMBOLIC : change dyn-link behavior [ig]
		break;
	case 17:		//DT_REL : val = addr of reloc table [req]
		info->rel_addr = dyn->val;
		break;
	case 18:		//DT_RELSZ : size of DT_REL table [req]
		info->rel_size = dyn->val;
		break;
	case 19:		//DT_RELENT : size of DT_REL entry [req]
		info->rel_ent = dyn->val;
		break;
	case 20:		//DT_PLTREL : val = type of PLT relocs (REL/RELA) [opt]
		info->plt_type = dyn->val;
		break;
	case 21:		//DT_DEBUG [opt]
		info->debug = dyn->val;
		break;
	case 22:		//DT_TEXTREL : relocs present in .text [opt]
		info->text_rel = dyn->val;
		break;
	case 23:		//DT_JMPREL : val = addr of PLT relocs [opt]
		info->plt_rel = dyn->val;
		break;
	case 0x70000000:	//DT_LOPROC [ig]
		break;
	case 0x7FFFFFFF:	//DT_HIPROC [ig]
		break;
	default:
		break;
	}
	return (1);
}

/*  ----------------------------------------------------doDynamic( int offset )
   -- Parse .dynamic section
*/
int do_dynamic(struct ELF_PROG *phdr, struct ELF_HDR *hdr)
{
	struct DYNSTR hNEEDED = { 0 }, hRPATH = { 0 };
	struct ELF_DYN **dynt, *dyn;
	char *strtab;
	int x, count;
	int symtab, size;	/* file offset/size of sym table */

	/* set up linked lists */
	pNEEDED = &hNEEDED;
	pRPATH = &hRPATH;

	/* create table of _DYNAMIC entries */
	count = phdr->p_filesz / sizeof (struct ELF_DYN);
	dynt = create_dynt(phdr, count);
	memset(&opt->dynamic, 0, sizeof (struct ELF_DYNAMIC_INFO));

	for (x = 0; x < count; x++) {
		/* figure out each _DYNAMIC entry */
		add_dyn_info(&opt->dynamic, dynt[x]);
	}

	if (opt->dynamic.str_addr) {
		/* OK, so I re-use symtab here, so sue me */
		symtab = get_file_offset(opt->dynamic.str_addr, hdr);

		/* load string table */
		strtab = malloc(opt->dynamic.str_size);
		lseek(opt->fd, opt->base + symtab, SEEK_SET);
		read(opt->fd, strtab, opt->dynamic.str_size);
	}

	if (opt->dynamic.soname) {
		/* make note of .soname */
		x = (int) opt->dynamic.soname;
		opt->dynamic.soname = malloc(strlen(strtab + x));
		strcpy(opt->dynamic.soname, strtab + x);
		sprintf(buf, "\tTarget soname: %s\n", strtab + x);
		strncat(opt->str.note, buf, opt->str.n_len);
	}

	/* notify disassembler of all required libraries */
	pNEEDED = &hNEEDED;
	while (pNEEDED->next != NULL && strtab) {
		pNEEDED = pNEEDED->next;
		(*opt->lib_new) (strtab + pNEEDED->offset, 0, 0);
		sprintf(buf, "\tTarget requires %s\n",
			strtab + pNEEDED->offset);
		strncat(opt->str.note, buf, opt->str.n_len);
	}

	/* make note of LD_LIBRARY_PATH */
	pRPATH = &hRPATH;
	while (pRPATH->next != NULL && strtab) {
		pRPATH = pRPATH->next;
		opt->dynamic.rpath = realloc(opt->dynamic.rpath, 2 +
					     strlen(strtab + pRPATH->offset));
		strcat(opt->dynamic.rpath, strtab + pRPATH->offset);
		if (pRPATH->next)
			strcat(opt->dynamic.rpath, ":");
	}

	if (opt->dynamic.rpath) {
		sprintf(buf, "\tTarget RPATH %s\n", opt->dynamic.rpath);
		strncat(opt->str.note, buf, opt->str.n_len);
	}

	if (opt->dynamic.sym_addr) {
		symtab = get_file_offset(opt->dynamic.sym_addr, hdr);
		/* TODO: replace w/ num_buckets from ELF hash table */
		/* strtable usually follows symtable: in PH, size of sym ! known */
		size = opt->dynamic.str_addr - opt->dynamic.sym_addr;
		/* do something useful with the symbols */
		if (strtab) {
			do_symtab(symtab, size, strtab, &hNEEDED, hdr);
			free(strtab);
		}
	}

	/* add comments to header Notes  */
	if (opt->dynamic.plt_addr) {
		sprintf(buf, "PLT size: %08X reloc: %08X rva: %08X\n",
			opt->dynamic.plt_size, opt->dynamic.plt_rel,
			opt->dynamic.plt_addr);
		strncat(opt->str.note, buf, opt->str.n_len);
	}
	if (opt->dynamic.rel_addr) {
		sprintf(buf, "REL size: %08X entry size: %08X rva: %08X\n",
			opt->dynamic.rel_size, opt->dynamic.rel_ent,
			opt->dynamic.rel_addr);
		strncat(opt->str.note, buf, opt->str.n_len);
	}
	if (opt->dynamic.rela_addr) {
		sprintf(buf, "RELA size: %08X entry size: %08X rva: %08X\n",
			opt->dynamic.rela_size, opt->dynamic.rela_ent,
			opt->dynamic.rela_addr);
		strncat(opt->str.note, buf, opt->str.n_len);
	}
	/* free the linked lists if they have anything in them... */
	pNEEDED = hNEEDED.next;
	while (pNEEDED) {
		hNEEDED.next = pNEEDED->next;
		free(pNEEDED);
		pNEEDED = hNEEDED.next;
	};

	pRPATH = hRPATH.next;
	while (pRPATH) {
		hRPATH.next = pRPATH->next;
		free(pRPATH);
		pRPATH = hRPATH.next;
	};


	free_dynt(dynt, count);
	return (1);
}


/* ------------------------------------------------- SECTION HEADER TABLE */
struct ELF_SECT **create_sht(struct ELF_HDR *hdr)
{
	struct ELF_SECT **sht = NULL;
	int x;

	/* Make array of pointers to section headers */
	sht = calloc(sizeof (void *), hdr->e_shnum);
	for (x = 0; x < hdr->e_shnum; x++) {
		sht[x] = malloc(sizeof (struct ELF_SECT));
		if ( ! sht[x] ) return(sht);
		lseek(opt->fd, opt->base + hdr->e_shoff + 
			(hdr->e_shentsize * x), SEEK_SET);
		read(opt->fd, sht[x], sizeof (struct ELF_SECT));
	}
	return (sht);
}

int free_sht(struct ELF_SECT **sht, struct ELF_HDR *hdr)
{
	int x;

	if (!sht || !hdr)
		return (0);

	for (x = 0; x < hdr->e_shnum; x++) {
		if (sht[x])
			free(sht[x]);
	}
	free(sht);
	return (1);
}

/*  ----------------------------------------------------doSectionTable( ) */

int do_sht(struct ELF_HDR *hdr)
{
	int x, len, rv = 0;
	char *sh_name, *strtab_buf, pad[16];
	struct ELF_SECT *shdr = NULL, *strtab = NULL, *symtab = NULL;
	struct ELF_SECT **sht = NULL;

	if (! hdr || !hdr->e_shnum)
		return (0);
	if (! g_sht ) g_sht = create_sht(hdr);
	sht = g_sht;

	/* Load String Table */
	if (hdr->e_shstrndx) {
		shdr = sht[hdr->e_shstrndx];
		e_strtable = malloc(shdr->sh_size);
		e_strtab_size = shdr->sh_size;
		lseek(opt->fd, opt->base + shdr->sh_offset, SEEK_SET);
		read(opt->fd, e_strtable, shdr->sh_size);
	}

	/* foreach section header */
	for (x = 0; x < hdr->e_shnum; x++) {
		shdr = sht[x];

		if (! shdr )	continue;
		/* get section name from string table */
		if (hdr->e_shstrndx && shdr->sh_name < e_strtab_size)
			sh_name = e_strtable + shdr->sh_name;
		else
			sh_name = "(null)";

		/* create SECTION object based on section name */
		if (!strcmp(sh_name, ".text")) {
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1100 | 0x01);
			/* if .text exists, header is OK */
			rv = 1;

		} else if (!strncmp(sh_name, ".data", 5)) {
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1200 | 0x06);

		} else if (!strncmp(sh_name, ".rodata", 7)) {
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1200 | 0x04);

		} else if (!strncmp(sh_name, ".bss", 4)) {
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1200 | 0x06);

		} else if (!strcmp(sh_name, ".note") ||
			   !strcmp(sh_name, ".comment")) {
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1700 | 0x04);

		} else if (!strncmp(sh_name, ".dyn", 4) ||
			   !strcmp(sh_name, ".got") ||
			   !strncmp(sh_name, ".rel", 4)) {
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1800 | 0x04);

		} else if (!strcmp(sh_name, ".symtab")) {
			(*opt->sec_new) (".symtab", shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1600 | 0x04);
			symtab = shdr;

		} else if (!strcmp(sh_name, ".strtab")) {
			(*opt->sec_new) (".strtab", shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1600 | 0x04);
			strtab = shdr;

		} else if (!strcmp(sh_name, ".debug") ||
			   !strcmp(sh_name, ".line")) {
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1600 | 0x04);

		} else if (!strcmp(sh_name, ".fini") ||
			   !strcmp(sh_name, ".init")) {
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1000 | 0x04 | 0x01);
		//} else if ( !strcmp(sh_name, ".interp")) {
		} else if (sh_name[0] && shdr->sh_size) {	
			(*opt->sec_new) (sh_name, shdr->sh_addr, shdr->sh_size,
					 shdr->sh_offset, 0x1000 | 0x04 );
		}
			
		/* align section names for pretty-printing */
		memset(pad, ' ', 15);
		len = strlen(sh_name);
		if (len > 16)
			len = 16;
		if (len < 1)
			len = 1;
		pad[16 - len] = 0;


		sprintf(buf, " %s%s%s \t%s \tOffset %8X VA %08X Size %X\n",
			sh_name, pad, get_sh_flags(shdr->sh_flags),
			get_sh_type(shdr->sh_type), shdr->sh_offset,
			shdr->sh_addr, shdr->sh_size);
		strncat(opt->str.sht_str, buf, opt->str.s_len);
	}

	/* if a symbol and a string table were found, load symbols */
	if (strtab && symtab) {
		strtab_buf = malloc(strtab->sh_size);
		lseek(opt->fd, opt->base + strtab->sh_offset, SEEK_SET);
		read(opt->fd, strtab_buf, strtab->sh_size);
		do_symtab(symtab->sh_offset, symtab->sh_size, strtab_buf, NULL,
			  hdr);
		free(strtab_buf);
	}


	return (rv);
}


/* ------------------------------------------------- PROGRAM HEADER TABLE */
struct ELF_PROG **create_pht(struct ELF_HDR *hdr)
{
	struct ELF_PROG **pht = NULL;
	int x;

	/* Make array of pointers to section headers */
	pht = calloc(sizeof (void *), hdr->e_phnum);
	for (x = 0; x < hdr->e_phnum; x++) {
		pht[x] = malloc(sizeof (struct ELF_PROG));
		if ( ! pht[x] ) return(pht);
		lseek(opt->fd, opt->base + hdr->e_phoff + (hdr->e_phentsize * x), SEEK_SET);
		read(opt->fd, pht[x], sizeof (struct ELF_PROG));
	}
	return (pht);
}

int free_pht(struct ELF_PROG **pht, struct ELF_HDR *hdr)
{
	int x;

	if (!pht || !hdr)
		return (0);

	for (x = 0; x < hdr->e_phnum; x++) {
		if (pht[x])
			free(pht[x]);
	}
	free(pht);
	return (1);
}

/*  --------------------------------------------------doProgramHeaders( bool )
   -- Parse ELF Program Header table and add to HeaderStr
   -- If Sections have not been created, create them
*/
int do_pht(struct ELF_HDR *hdr, int create_sections)
{
	int x, pos, size;
	char progType[16], progFlags[4];
	char *noName = "(null)";
	struct ELF_PROG **pht = NULL, *phdr = NULL, *dyn_hdr = NULL;

	if (!hdr->e_phnum)
		return (0);
	if (! g_pht) g_pht = create_pht(hdr);
	pht = g_pht;


	for (x = 0; x < hdr->e_phnum; x++) {
		phdr = pht[x];

		/* set size of section */
		if (phdr->p_filesz > phdr->p_memsz)
			size = phdr->p_filesz;
		else
			size = phdr->p_memsz;

		/* is this the DYNAMIC section? */
		if (phdr->p_type == 2)
			dyn_hdr = phdr;

		/*  find entry point, create sections */
		if (create_sections) {
			strncat(opt->str.note,
				"Using Program Headers to determine "
				".text section", opt->str.n_len);

			/* if entry point is in this section ... */
			if (hdr->e_entry >= phdr->p_vaddr &&
			    hdr->e_entry < phdr->p_vaddr + phdr->p_filesz) {
				/* Create .text section */
				(*opt->sec_new) (".text", phdr->p_vaddr, size, phdr->p_offset,
						 0x1100 | 0x01 | 0x04);
			} else if (phdr->p_type == 1 && phdr->p_flags & 2) {
				/* this section contains writeable data */
				(*opt->sec_new) (".data", phdr->p_vaddr, size, phdr->p_offset,
						 0x1200 | 0x02 | 0x04);
			} else if (phdr->p_type == 1 && phdr->p_flags & 4) {
				/* this section containes read-only data */
				(*opt->sec_new) (".rodata", phdr->p_vaddr, size,
						 phdr->p_offset, 0x1200 | 0x04);
			}
		}
		sprintf(buf, " %d: \t%s \t%s Offset %8X \tVA %08X Size %X\n", x,
			get_ph_type(phdr->p_type), get_ph_flags(phdr->p_flags),
			phdr->p_offset, phdr->p_vaddr, size);
		strncat(opt->str.pht_str, buf, opt->str.p_len);

	}

	/* do dynamic header last */
	if (dyn_hdr) {
		do_dynamic(dyn_hdr, hdr);
	}

	return (0);
}

/* ------------------------------------------------------- Data Types */
int do_elf_datatypes()
{
	int e_struct, d1, d2, d3, d4, d5, d6;

	/* create data structures */
	d1 = dtype_new("Elf32_Addr", 4, DT_UNSIGNED);
	d2 = dtype_new("Elf32_Half", 2, DT_UNSIGNED);
	d3 = dtype_new("Elf32_Off", 4, DT_UNSIGNED);
	d4 = dtype_new("Elf32_Sword", 4, DT_SIGNED);
	d5 = dtype_new("Elf32_Word", 4, DT_UNSIGNED);
	d6 = dtype_get("unsigned char");
	if (! d6 ) {
		d6 = dtype_new("unsigned char", 1, DT_UNSIGNED);
	}

	/* create ELF Header struct */
	e_struct = struct_new("Elf32_Ehdr", 52);
	/*                  struct  type sz  order name */
	struct_add_member(e_struct, d6, 16, 1, "e_ident");
	struct_add_member(e_struct, d2, 1, 2, "e_type");
	struct_add_member(e_struct, d2, 1, 3, "e_machine");
	struct_add_member(e_struct, d5, 1, 4, "e_version");
	struct_add_member(e_struct, d1, 1, 5, "e_entry");
	struct_add_member(e_struct, d3, 1, 6, "e_phoff");
	struct_add_member(e_struct, d3, 1, 7, "e_shoff");
	struct_add_member(e_struct, d5, 1, 8, "e_flags");
	struct_add_member(e_struct, d2, 1, 9, "e_ehsize");
	struct_add_member(e_struct, d2, 1, 10, "e_phentsize");
	struct_add_member(e_struct, d2, 1, 11, "e_phnum");
	struct_add_member(e_struct, d2, 1, 12, "e_shentsize");
	struct_add_member(e_struct, d2, 1, 13, "e_shsnum");
	struct_add_member(e_struct, d2, 1, 14, "e_shstrndx");

	/* apply this struct to the file itself */
	(*opt->addr_new) (0, 52, 0, ADDR_STRUCT | ADDR_DATA);
	(*opt->struct_apply)( 0, e_struct );
	return (1);
}



/*  -------------------------------------------------------------------main()
*/
int parse_elf(struct ELF_INFO *info)
{
	unsigned long e_entry;
	struct ELF_HDR *h;
	struct ELF_IDENT *ident;

	if (!info)
		return (EBADPARAM);
	opt = info;
	h = &opt->header;
	/* clear globals in case we are called more than once :) */
	g_sht = NULL;
	g_pht = NULL;
	e_strtable = NULL;

	/*  ______Load the ELF File Header_______ */
	lseek( opt->fd, opt->base, SEEK_SET);
	if( read(opt->fd, h, sizeof (struct ELF_HDR)) 
			< sizeof (struct ELF_HDR) ) {
		return(0);
	}

	/* Check Magic Number */
	if (strncmp(h->e_ident, strMagic, 4)) {
		return (EBADELF);
	}

	/* create section for header */
	(*opt->sec_new) ("header", 0, h->e_ehsize, 0, 0x1000);

	/* start the header strings */
	ident = (struct ELF_IDENT *) h;
	sprintf(buf, "File Format: ELF %s %s\tType: %s\tMachine: %s\n"
		"Entry: %08X\tSection Table: %08X\tProgram Header: %08X\n",
		get_elf_class(ident->ei_class), get_elf_data(ident->ei_data),
		get_elf_type(h->e_type), get_mach_name(h->e_machine),
		h->e_entry, h->e_shoff, h->e_phoff);
	strncat(opt->str.elf_str, buf, opt->str.e_len);

	/* if section headers do not exist, use program headers for .text etc */
	if (do_sht(h)) {
		do_pht(h, 0);
	} else {
		do_pht(h, 1);
	}

	if (g_sht) free_sht(g_sht, h);
	if (g_pht) free_pht(g_pht, h);
	return (1);
}
