#include <dlfcn.h>
#include <glob.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <extension.h>
#include <debug.h>
/* dammit ... had to add these for the IN_EIC error-trap flag */
#include <bastard.h>
extern struct DISASM_ENV disasm_env;
struct EXT__ARCH *ext_arch = NULL;
struct EXT__ASM *ext_asm = NULL;
struct EXT__COMP *ext_comp = NULL;
struct EXT__FORMAT  *ext_format = NULL;
struct EXT__LANG	*ext_lang = NULL;
struct EXT__OS *ext_os = NULL;
struct EXT__PLUGIN *ext_plugin = NULL, *ext_plugin_list = NULL;
struct EXT__ENGINE *ext_engine = NULL;
/* </DAMMIT> */

int ext_plugin_exists( char *filename ); /* forward decl */

/* the way it works:
 * Extensions are loaded and unloaded by referring to their type constants.
 * Only routines which call the extensions will refer to the extension
 * structures directly, so there is no API for doing do.
 *
 * Note that the functions each extension provides will be callable via
 * the bastard API, so scripts will not call the extensions directly.
 */
void ext_fprint_versions( FILE *f ) {
	struct EXT__PLUGIN *plugin;

	if ( ! ext_arch || ! ext_asm || ! ext_comp ||
		! ext_format || ! ext_lang || !ext_os   )
		return;

	fprintf( f, "\text_arch %s: %2.2f\n", ext_arch->ext.filename, 
			ext_arch->ext.version );
	fprintf( f, "\text_asm %s: %2.2f\n", ext_asm->ext.filename, 
			ext_asm->ext.version );
	fprintf( f, "\text_comp %s: %2.2f\n", ext_comp->ext.filename, 
			ext_comp->ext.version );
	fprintf( f, "\text_format %s: %2.2f\n", ext_format->ext.filename, 
			ext_format->ext.version );
	fprintf( f, "\text_lang %s: %2.2f\n", ext_lang->ext.filename, 
			ext_lang->ext.version );
	fprintf( f, "\text_os %s: %2.2f\n", ext_os->ext.filename, 
			ext_os->ext.version );

	if ( ext_plugin_list ) {
		plugin = ext_plugin_list->next;
		while ( plugin ) {
			fprintf(f, "\tplugin %s: %2.2f\n", plugin->ext.filename,
				plugin->ext.version );
			plugin = plugin->next;
		}
	}
	return;
}
int ext_gen_filename( char *ext, char *name, char *buf, int buf_len ) {
	struct stat statbuf;
	snprintf( buf, buf_len, "%s/%s/%s.bc", env_get_share(), ext,
		name );
	if (stat(buf, &statbuf)) {
		snprintf( buf, buf_len, "%s/%s/lib%s.so", env_get_share(),
			ext, name );
		if (stat(buf, &statbuf)) {
			return(sys_set_lasterr(4594));
		}
	}
	return( strlen(buf) );
}

void sys_init_extensions()
{
	ext_arch = calloc(sizeof (struct EXT__ARCH), 1);
	ext_asm = calloc(sizeof (struct EXT__ASM), 1);
	ext_format = calloc(sizeof (struct EXT__FORMAT), 1);
	ext_lang = calloc(sizeof (struct EXT__LANG), 1);
	ext_comp = calloc(sizeof (struct EXT__COMP), 1);
	ext_os = calloc(sizeof (struct EXT__OS), 1);
	ext_engine = calloc(sizeof (struct EXT__ENGINE), 1);
	ext_plugin = ext_plugin_list = NULL;

	return;
}

void cleanup_extensions()
{
	struct EXT__PLUGIN *ptr;
	/* prevent attempts to free() twice */
	if ( ! ext_arch ) 	return;
	/* foreach extension, call cleanup and unload */
	unload_extension(EXT_ARCH);
	unload_extension(EXT_ASM);
	unload_extension(EXT_FORMAT);
	unload_extension(EXT_LANG);
	unload_extension(EXT_COMP);
	unload_extension(EXT_OS);
	unload_extension(EXT_ENGINE);

	/* free the demn memory */
	free(ext_arch);
	free(ext_asm);
	free(ext_format);
	free(ext_lang);
	free(ext_comp);
	free(ext_os);
	free(ext_engine);

	/* cleanup plugins */
/*	if ( ext_plugin_list ) {
		ptr = ext_plugin_list->next;
		while ( ptr ) {
			ext_plugin_unload( ptr->name );
			ptr = ext_plugin_list->next;
		}
		ext_plugin_unload( ext_plugin_list->name );
	}
*/	
	ext_arch = NULL; ext_asm = NULL; ext_format = NULL;
	ext_lang = NULL; ext_comp = NULL; ext_os = NULL;
	ext_engine = NULL; ext_plugin = NULL;

	return;
}

static int loadext_internal(struct EXTENSION *ext, int type, char *filename,
		     int method)
{

	/* if previously loaded, just return */
	/* NO!! this prevents us from changing options!
	DEBUG_PrintMsg("checking if %s is already loaded\n", filename);
	if (ext->filename && !strcmp(filename, ext->filename))
		return (1);
	*/
	if ( type == EXT_PLUGIN  && ext_plugin_exists( filename ) ){
		return(3);
	}
	/* if another extension of the same type is currently loaded, unload it */
	/* NOTE: disabling this is a total hack to get libMAGIC working
	 *       [since libMAGIC replaces itself with another extension]
	 *       For now the hack is [wisely] limitged to EXT_FORMAT.
	 *            if (ext->filename || ext->lib)
	 */
	if ((ext->filename || ext->lib) && type != EXT_FORMAT)
		unload_extension(type);

	/* if a library, load it */
	if (method == EXT_SO)
		if (!(ext->lib = dlopen(filename, RTLD_NOW))) {
			sys_errmsg(4200, dlerror());
			return (sys_set_lasterr(4200));
		}
	return (2);
}

int load_extension(int type, char *filename, void *param)
{
	int len, method, rv;	/* name length, script or binary */
	struct EXTENSION *ext;

	/* determine extension type based on file extension */
	len = strlen(filename);
	DEBUG_PrintMsg("Starting to Load %s\n", filename);
	if (!strncmp(".bc", &filename[len - 3], 3))
		method = EXT_BC;	/* we have a script */
	else {			/* binary extension is therefore arbitrary */
		method = EXT_SO;
	}

	/* set up function pointers for each extension */
	switch (type) {
	case EXT_PLUGIN:
		ext = &ext_plugin->ext;
		if ((rv = loadext_internal(ext, type, filename, method)) != 2)
			return (rv);
		DEBUG_PrintMsg("loading plugin %s\n", filename);
		if (method == EXT_BC) {
			ext->fn_init = NULL;
			ext->fn_cleanup = NULL;
			ext_plugin->fn_main = &plugin_main_stub;
		} else {	/* method == EXT_SO */
			ext->fn_init = dlsym(ext->lib, "ext_plugin_init");
			ext->fn_cleanup = dlsym(ext->lib, "ext_plugin_cleanup");
			ext_plugin->fn_main = dlsym(ext->lib, "plugin_main");
		}
		break;
	case EXT_ENGINE:
		ext = &ext_engine->ext;
		if ((rv = loadext_internal(ext, type, filename, method)) != 2)
			return (rv);
		DEBUG_PrintMsg("loading engine %s\n", filename);
		if (method == EXT_BC) {
			ext->fn_init = NULL;
			ext->fn_cleanup = NULL;
			ext_engine->fn_main = &engine_main_stub;
		} else {	/* method == EXT_SO */
			ext->fn_init = dlsym(ext->lib, "ext_engine_init");
			ext->fn_cleanup = dlsym(ext->lib, "ext_engine_cleanup");
			ext_engine->fn_main = dlsym(ext->lib, "engine_main");
		}
		break;
	case EXT_FORMAT:
		ext = &ext_format->ext;
		if ((rv = loadext_internal(ext, type, filename, method)) != 2)
			return (rv);
		DEBUG_PrintMsg("loading format %s\n", filename);
		if (method == EXT_BC) {
			ext->fn_init = NULL;
			ext->fn_cleanup = NULL;
			ext_format->fn_read_header = &read_header_stub;
		} else {	/* method == EXT_SO */
			ext->fn_init = dlsym(ext->lib, "ext_format_init");
			ext->fn_cleanup = dlsym(ext->lib, "ext_format_cleanup");
			ext_format->fn_read_header =
			    dlsym(ext->lib, "read_header");
			if (!ext_format->fn_read_header)
				method = 0;
		}
		break;
	case EXT_ARCH:
		ext = &ext_arch->ext;
		if ((rv = loadext_internal(ext, type, filename, method)) != 2)
			return (rv);
		DEBUG_PrintMsg("loading arch %s\n", filename);
		if (method == EXT_BC) {
			ext->fn_init = &arch_init_stub;
			ext->fn_cleanup = &arch_cleanup_stub;
			ext_arch->fn_disasm_addr = &disasm_addr_stub;
			ext_arch->fn_disasm_inv = &disasm_inv_stub;
			ext_arch->fn_code_pat = &code_pat_stub;
			ext_arch->fn_gen_effect = &gen_effect_stub;
			ext_arch->fn_gen_int = &gen_int_stub;
		} else {	/* method == EXT_SO */
			ext->fn_init = dlsym(ext->lib, "ext_arch_init");
			ext->fn_cleanup = dlsym(ext->lib, "ext_arch_cleanup");
			ext_arch->fn_disasm_addr =
			    dlsym(ext->lib, "disasm_addr");
			ext_arch->fn_disasm_inv =
			    dlsym(ext->lib, "disasm_invariant");
			ext_arch->fn_code_pat =
			    dlsym(ext->lib, "test_for_code_pattern");
			ext_arch->fn_gen_effect =
			    dlsym(ext->lib, "gen_code_effect");
			ext_arch->fn_gen_int = dlsym(ext->lib, "gen_int");
			if (!ext->fn_init || !ext_arch->fn_disasm_addr)
				method = 0;
		}
		break;
	case EXT_ASM:
		ext = &ext_asm->ext;
		if ((rv = loadext_internal(ext, type, filename, method)) != 2)
			return (rv);
		DEBUG_PrintMsg("loading asm %s\n", filename);
		if (method == EXT_BC) {
			ext->fn_init = &asm_init_stub;
			ext->fn_cleanup = &asm_cleanup_stub;
			ext_asm->fn_sprint_code = &sprint_code_stub;
			ext_asm->fn_sprint_func_s = &sprint_func_start_stub;
			ext_asm->fn_sprint_func_e = &sprint_func_end_stub;
			ext_asm->fn_sprint_struct = &sprint_struct_stub;
			ext_asm->fn_sprint_sec_s = &sprint_section_start_stub;
			ext_asm->fn_sprint_sec_e = &sprint_section_end_stub;
			ext_asm->fn_sprint_addrexp = &sprint_addrexp_stub;
			ext_asm->fn_gen_file = &gen_asmfile_stub;
		} else {	/* method == EXT_SO */
			ext->fn_init = dlsym(ext->lib, "ext_asm_init");
			ext->fn_cleanup = dlsym(ext->lib, "ext_asm_cleanup");
			ext_asm->fn_sprint_code =
			    dlsym(ext->lib, "sprint_code");
			ext_asm->fn_sprint_struct =
			    dlsym(ext->lib, "sprint_asm_struct");
			ext_asm->fn_sprint_func_s =
			    dlsym(ext->lib, "sprint_asm_func_start");
			ext_asm->fn_sprint_func_e =
			    dlsym(ext->lib, "sprint_asm_func_end");
			ext_asm->fn_sprint_sec_s =
			    dlsym(ext->lib, "sprint_section_start");
			ext_asm->fn_sprint_sec_e =
			    dlsym(ext->lib, "sprint_section_end");
			ext_asm->fn_sprint_addrexp =
			    dlsym(ext->lib, "sprint_addrexp");
			ext_asm->fn_gen_file = dlsym(ext->lib, "gen_asm_file");
			if (!ext->fn_init || !ext_asm->fn_sprint_code ||
			    !ext_asm->fn_sprint_addrexp) method = 0;
		}
		break;
	case EXT_LANG:
		ext = &ext_lang->ext;
		if ((rv = loadext_internal(ext, type, filename, method)) != 2)
			return (rv);
		DEBUG_PrintMsg("loading lang %s\n", filename);
		if (method == EXT_BC) {
			ext->fn_init = &lang_init_stub;
			ext->fn_cleanup = &lang_cleanup_stub;
			ext_lang->fn_sprint_proto = &sprint_proto_stub;
			ext_lang->fn_sprint_func = &sprint_func_stub;
			ext_lang->fn_gen_final = &gen_final_stub;
			ext_lang->fn_gen_file = &gen_hllfile_stub;
		} else {	/* method == EXT_SO */
			ext->fn_init = dlsym(ext->lib, "ext_lang_init");
			ext->fn_cleanup = dlsym(ext->lib, "ext_lang_cleanup");
			ext_lang->fn_sprint_proto =
			    dlsym(ext->lib, "sprint_proto");
			ext_lang->fn_sprint_func =
			    dlsym(ext->lib, "sprint_func");
			ext_lang->fn_gen_final = dlsym(ext->lib, "gen_final");
			ext_lang->fn_gen_file = dlsym(ext->lib, "gen_file");
		}
		break;
	case EXT_COMP:
		ext = &ext_comp->ext;
		if ((rv = loadext_internal(ext, type, filename, method)) != 2)
			return (rv);
		DEBUG_PrintMsg("loading comp %s\n", filename);
		if (method == EXT_BC) {
			ext->fn_init = &comp_init_stub;
			ext->fn_cleanup = &comp_cleanup_stub;
			ext_comp->fn_findstr = (findstr_fn) & findstr_stub;
			ext_comp->fn_makestr = (makestr_fn) & makestr_stub;
			ext_comp->fn_datatype = (datatype_fn) & datatype_stub;
			ext_comp->fn_genargs = (genargs_fn) & genargs_stub;
			ext_comp->fn_effect = (genargs_fn) & func_effect_stub;
		} else {	/* method == EXT_SO */
			ext->fn_init = dlsym(ext->lib, "ext_comp_init");
			ext->fn_cleanup = dlsym(ext->lib, "ext_comp_cleanup");
			ext_comp->fn_findstr = dlsym(ext->lib, "findstr");
			ext_comp->fn_makestr = dlsym(ext->lib, "makestr");
			ext_comp->fn_datatype = dlsym(ext->lib, "add_data_types");
			ext_comp->fn_genargs = dlsym(ext->lib, "comp_func_genargs");
			ext_comp->fn_effect=dlsym(ext->lib,"comp_func_effect");
		}
		break;
	case EXT_OS:
		ext = &ext_os->ext;
		if ((rv = loadext_internal(ext, type, filename, method)) != 2)
			return (rv);
		DEBUG_PrintMsg("loading os %s\n", filename);
		if (method == EXT_BC) {
			ext->fn_init = &os_init_stub;
			ext->fn_cleanup = &os_cleanup_stub;
			ext_os->fn_sysref_gen = (sysref_gen_fn) &sysref_gen_stub;
			ext_os->fn_sysref_name = (sysref_name_fn) &sysref_name_stub;
		} else {	/* method == EXT_SO */
			ext->fn_init = dlsym(ext->lib, "ext_os_init");
			ext->fn_cleanup = dlsym(ext->lib, "ext_os_cleanup");
			ext_os->fn_sysref_gen = dlsym(ext->lib, "os_sysref_gen");
			ext_os->fn_sysref_name = dlsym(ext->lib, "os_sysref_name");
		}
		break;
	default:
		return (0);	/* this is an unknown type */
	};

	/* set filename, etc for extension */
	ext->filename = calloc(len + 1, 1);
	strncpy(ext->filename, filename, len);
	ext->flags = method | type | EXT_OPEN;
	ext->version = BASTARD_VERSION;	/* default version number */

	/* if init call init */
	DEBUG_PrintMsg("Calling init for %s\n", filename);
	if (ext->fn_init)
		ext->fn_init(param);
	ext->flags = (ext->flags & ~EXT_STATUS_MASK) | EXT_INIT;
	DEBUG_PrintMsg("Done Loading %s\n", filename);
	return (1);
}

struct EXTENSION *get_ext_struct(int type)
{
	switch (type) {
	case EXT_PLUGIN:
		return (&ext_plugin->ext);
	case EXT_ENGINE:
		return (&ext_engine->ext);
	case EXT_FORMAT:
		return (&ext_format->ext);
	case EXT_ARCH:
		return (&ext_arch->ext);
	case EXT_ASM:
		return (&ext_asm->ext);
	case EXT_LANG:
		return (&ext_lang->ext);
	case EXT_COMP:
		return (&ext_comp->ext);
	case EXT_OS:
		return (&ext_os->ext);
	default:
		break;
	}
	return (NULL);
}

int get_extension_flags(int type)
{
	struct EXTENSION *ext = get_ext_struct(type);
	if (!ext)
		return (0);
	return (ext->flags);
}

int unload_extension(int type)
{
	struct EXTENSION *ext = get_ext_struct(type);
	if (!ext)
		return (0);
	/* if cleanup call cleanup */
	DEBUG_PrintMsg("Unloading extension %s\n", ext->filename);
	ext->flags = (ext->flags & ~EXT_STATUS_MASK) | EXT_CLEANUP;
	if (ext->fn_cleanup)
		ext->fn_cleanup();
	free(ext->filename);
	if (ext->lib)
		dlclose(ext->lib);
	ext->flags = (ext->flags & ~EXT_STATUS_MASK) | EXT_CLOSED;
	memset(ext, 0, sizeof (struct EXTENSION));
	return (1);
}

/* plugin routines */
static int ext_plugin_name( struct EXT__PLUGIN *plug, char *filename ){
	char *ptr;
	ptr = strrchr(filename, '/');
	if ( ptr ) 
		ptr++;
	else
		ptr = filename;
	strncpy( plug->name, ptr, 63 );
	return( strlen(plug->name) );
}

static struct EXT__PLUGIN *ext_new_plugin( void) {
	struct EXT__PLUGIN *plug, *ptr;
	plug = calloc(sizeof (struct EXT__PLUGIN), 1);
	if ( plug ) {
		/* insert into linked list */
		plug->next = ext_plugin_list;
		if ( ext_plugin_list ) 
			ext_plugin_list->prev = plug;
		/* set current plugin to new one */
		ext_plugin = ext_plugin_list = plug;
	}
	return( plug );
}

static struct EXT__PLUGIN * ext_plugin_find_filename( char *filename ) {
	struct EXTENSION *ext;
	struct EXT__PLUGIN *ptr;
	ptr = ext_plugin_list;
	while ( ptr ) {
		ext = &ptr->ext;
		if ( ext->filename && ! strcmp( ext->filename, filename ) )
			return( ptr );
		ptr = ptr->next;
	}
	return(NULL);
}

int ext_plugin_exists( char *filename ) {
	struct EXT__PLUGIN *ptr;
	ptr = ext_plugin_find_filename( filename );
	if ( ptr )
		return(1);
	return(0);
}

char * ext_plugin_load( char *filename, int options ) {
	struct EXT__PLUGIN *plug;
	plug = ext_plugin_find_filename( filename );
	if (! plug ) {
		plug = ext_new_plugin( );
		ext_plugin_name( plug, filename );
		plug->options = options;
		/* ext_plugin now set to new plugin; pass 'plug' as param */
		load_extension( EXT_PLUGIN, filename, plug );
	} 
		
	return( plug->name );
}

static struct EXT__PLUGIN *ext_find_plugin( char *name ) {
	struct EXT__PLUGIN *ptr;
	ptr = ext_plugin_list;
	while ( ptr ) {
		if ( ! strncmp( ptr->name, name, 64 ) )
			return( ptr );
		ptr = ptr->next;
	}
	return(NULL);
}

static int ext_plugin_change( char *name ) {
	/* change current plugin */
	struct EXT__PLUGIN * plug;
	plug = ext_find_plugin( name );
	if ( plug ) {
		ext_plugin = plug;
		return(1);
	}
	return(0);
}

int ext_plugin_unload( char *name ) {
	struct EXT__PLUGIN * plug, *ptr;
	struct EXTENSION *ext;

	plug = ext_find_plugin( name );
	if ( ! plug )	return(0);
	ext = &ext_plugin->ext;

	/* change current plugin */
	if ( ext_plugin == plug )
		ext_plugin = plug->next;
	/* remove from list */
	if ( plug == ext_plugin_list ) {
		ext_plugin_list = plug->next;
	} else {
		plug->prev->next = plug->next;
		plug->next->prev = plug->prev;
	}
	/* perform cleanup */
	if (ext->fn_cleanup)
		ext->fn_cleanup();
	free(ext->filename);
	if (ext->lib)
		dlclose(ext->lib);
	free( plug );
	return(1);
}

/* wrappers ------------- */
/* these are used call .bc script routines instead of .so routines */

/* PLUGIN stubs */
int plugin_main_stub(void *plugin_arg)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_plugin->ext.filename);
	DEBUG_PrintMsg("executing plugin for  %s\n", ext_plugin->ext.filename);

	EiC_parseString("void * unsafe plugin_arg @ %ld;", (long) plugin_arg);
	if (plugin_arg)
		EiC_parseString("plugin_main(plugin_arg);");
	else
		EiC_parseString("plugin_main(NULL);");

	EiCp_ResetStart();
	return (1);
}

/* ENGINE stubs */
int engine_main_stub(void *engine_arg)
{
	disasm_env.flags &= IN_EIC;
	EiC_parseString(":reset here");
	sys_load_script(ext_engine->ext.filename);
	DEBUG_PrintMsg("executing engine for  %s\n", ext_engine->ext.filename);

	EiC_parseString("void * unsafe engine_arg @ %ld;", (long) engine_arg);
	if (engine_arg)
		EiC_parseString("engine_main(engine_arg);");
	else
		EiC_parseString("engine_main(NULL);");


	EiCp_ResetStart();
	disasm_env.flags &= ~IN_EIC;
	return (1);
}

/* FORMAT stubs */
int read_header_stub(void)
{
	int retval = 0;

	EiC_parseString(":reset here");
	sys_load_script(ext_format->ext.filename);
	DEBUG_PrintMsg("executing read_header for  %s\n",
		       ext_format->ext.filename);

	EiC_parseString("int rv @ %ld;", (long) &retval);
	EiC_parseString("rv = read_header();");
	EiCp_ResetStart();
	//sprintf(cmd, ":clear \"%s\"\n", script);
	return (retval);
}


/* ARCH stubs */
void arch_init_stub(void *param)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_arch->ext.filename);
	EiC_parseString("void * unsafe arch_init_param @ %ld;", (long) param);
	EiC_parseString("arch_init(arch_init_param);");
	EiCp_ResetStart();
	return;
}
void arch_cleanup_stub(void)
{
	EiC_parseString(":reset here");
	EiC_parseString("arch_cleanup( NULL);");
	EiCp_ResetStart();
	return;
}

int disasm_addr_stub(unsigned char *c, int n, struct code *inst, long rva)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_arch->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_arch->ext.filename);

	EiC_parseString("unsigned char * unsafe c @ %ld;", (long) c);
	EiC_parseString("int n @ %ld;", (long) &n);
	EiC_parseString("struct code * unsafe inst @ %ld;", (long) inst);
	EiC_parseString("unsigned int rva @ %ld;", (long) &rva);
	EiC_parseString("disasm_addr(c, n, inst, rva);");

	EiCp_ResetStart();
	return (1);
}

int disasm_inv_stub(unsigned char *in, int in_len, struct ARCH_INVARIANT *inv)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_arch->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_arch->ext.filename);

	EiC_parseString("unsigned char * unsafe c @ %ld;", (long) in);
	EiC_parseString("int n @ %ld;", (long) &in_len);
	EiC_parseString("struct ARCH_INVARIANT * unsafe inv @ %ld;", (long) inv);
	EiC_parseString("disasm_invariant(in, in_len, inv);");

	EiCp_ResetStart();
	return (1);
}

int code_pat_stub(unsigned long rva, int pattern)
{
	int rv;

	EiC_parseString(":reset here");
	sys_load_script(ext_arch->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_arch->ext.filename);

	EiC_parseString("unsigned long rva @ %ld;", (long) rva);
	EiC_parseString("int pattern @ %ld;", (long) pattern);
	EiC_parseString("int rv @ %ld;", (long) rv);
	EiC_parseString("rv = test_code_for_pattern(rva, pattern);");

	EiCp_ResetStart();
	return (rv);
}
int gen_effect_stub(struct code *c, struct code_effect *e)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_arch->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_arch->ext.filename);

	EiC_parseString("struct code * unsafe c @ %ld;", (long) c);
	EiC_parseString("struct code_effect * unsafe e @ %ld;", (long) e);
	EiC_parseString("gen_code_effect(c, e);");

	EiCp_ResetStart();
	return (1);
}
int gen_int_stub(struct function *f )
{
	EiC_parseString(":reset here");
	sys_load_script(ext_arch->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_arch->ext.filename);

	EiC_parseString("struct function * unsafe f @ %ld;", (long) f);
	EiC_parseString("gen_int(f);");

	EiCp_ResetStart();
	return (1);
}

/* ASM stubs */
void asm_init_stub(void *param)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	EiC_parseString("void * unsafe asm_init_param @ %ld;", (long) param);
	EiC_parseString("asm_init(asm_init_param);");
	EiCp_ResetStart();
	return;
}
void asm_cleanup_stub(void)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	EiC_parseString("asm_cleanup( NULL);");
	EiCp_ResetStart();
}
int sprint_code_stub(long rva, char *buf, int len, int output)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_asm->ext.filename);

	EiC_parseString("long rva @ %ld;", (long) &rva);
	EiC_parseString("char * unsafe buf @ %ld;", (long) buf);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("int output @ %ld;", (long) &output);
	EiC_parseString("sprint_code(rva, buf, len, output);");

	EiCp_ResetStart();
	return (1);
}
int sprint_func_start_stub(char *str, int len, int func_id)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_asm->ext.filename);

	EiC_parseString("char * unsafe str @ %ld;", (long) str);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("int func_id @ %ld;", (long) &func_id);
	EiC_parseString("sprint_func_start(rvstr, len, func_id);");

	EiCp_ResetStart();
	return (1);
}
int sprint_func_end_stub(char *str, int len, int func_id)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_asm->ext.filename);

	EiC_parseString("char * unsafe str @ %ld;", (long) str);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("int func_id @ %ld;", (long) &func_id);
	EiC_parseString("sprint_func_end(rvstr, len, func_id);");

	EiCp_ResetStart();
	return (1);
}
int sprint_struct_stub(char *str, int len, long rva)
{
	char cmd[2048], *script;

	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_asm->ext.filename);

	EiC_parseString("char * unsafe str @ %ld;", (long) str);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("int rva @ %ld;", (long) &rva);
	EiC_parseString("sprint_struct(str, len, rva);");

	EiCp_ResetStart();
	return (1);
}
int sprint_section_start_stub(char *str, int len, char *name)
{
	char cmd[2048], *script;

	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_asm->ext.filename);

	EiC_parseString("char * unsafe str @ %ld;", (long) str);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("char * unsafe name @ %ld;", (long) name);
	EiC_parseString("sprint_section_start(str, len, name);");

	EiCp_ResetStart();
	return (1);
}
int sprint_section_end_stub(char *str, int len, char *name)
{
	char cmd[2048], *script;

	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_asm->ext.filename);

	EiC_parseString("char * unsafe str @ %ld;", (long) str);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("char * unsafe name @ %ld;", (long) name);
	EiC_parseString("sprint_section_end(str, len, name);");

	EiCp_ResetStart();
	return (1);
}

int gen_asmfile_stub(FILE *f)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_asm->ext.filename);

	EiC_parseString("FILE * unsafe f @ %ld;", (long) f);
	EiC_parseString("gen_file(f);");

	EiCp_ResetStart();
	return (1);
}

int sprint_addrexp_stub(char *buf, int len, char *scale, char *index,
			char *base, char *disp, int sign)
{
	char cmd[2048], *script;

	EiC_parseString(":reset here");
	sys_load_script(ext_asm->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_asm->ext.filename);


	EiC_parseString("char * unsafe buf @ %ld;", (long) buf);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("char * unsafe scale @ %ld;", (long) scale);
	EiC_parseString("char * unsafe index @ %ld;", (long) index);
	EiC_parseString("char * unsafe base @ %ld;", (long) base);
	EiC_parseString("char * unsafe disp @ %ld;", (long) disp);
	EiC_parseString("int sign @ %ld;", (long) &sign);
	EiC_parseString
	    ("sprint_addrexp(buf, len, scale, index, base, disp, sign);");

	EiCp_ResetStart();
	return (1);
}

/* OS stubs */
void os_init_stub(void *param)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_os->ext.filename);
	EiC_parseString("void * unsafe os_init_param @ %ld;", (long) param);
	EiC_parseString("os_init(os_init_param);");
	EiCp_ResetStart();
	return;
}
void os_cleanup_stub(void)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_os->ext.filename);
	EiC_parseString("os_cleanup( NULL);");
	EiCp_ResetStart();
	return;
}

void sysref_gen_stub( struct code *c, struct sysref *s) {
	EiC_parseString(":reset here");
	sys_load_script(ext_os->ext.filename);
	EiC_parseString("struct code * unsafe c @ %ld;", (long) s);
	EiC_parseString("struct sysref * unsafe s @ %ld;", (long) c);
	EiC_parseString("os_sysref_gen(c, s);");
	EiCp_ResetStart();
	return;
}
void sysref_name_stub( struct sysref *s, char *buf, int len) {
	EiC_parseString(":reset here");
	sys_load_script(ext_os->ext.filename);
	EiC_parseString("struct sysref * unsafe s @ %ld;", (long) s);
	EiC_parseString("char * unsafe buf @ %ld;", (long) buf);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("os_sysref_name(s, buf, len);");
	EiCp_ResetStart();
	return;
}
/* COMP stubs */
void comp_init_stub(void *param)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_comp->ext.filename);
	EiC_parseString("void * unsafe comp_init_param @ %ld;", (long) param);
	EiC_parseString("comp_init(comp_init_param);");
	EiCp_ResetStart();
	return;
}
void comp_cleanup_stub(void)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_comp->ext.filename);
	EiC_parseString("comp_cleanup( NULL);");
	EiCp_ResetStart();
	return;
}
int findstr_stub(char *buf, int len, int *pos, char *text, int text_len, 
			int type)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_comp->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("char * unsafe buf @ %ld;", (long) buf);
	EiC_parseString("char * unsafe text @ %ld;", (long) text);
	EiC_parseString("int * unsafe pos @ %ld;", (long) pos);
	EiC_parseString("int text_len @ %ld;", (long) &text_len);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("int type @ %ld;", (long) &type);
	EiC_parseString("findstr(buf, len, pos, text, text_len, type);");
	EiCp_ResetStart();
	return (1);
}
int datatype_stub(int mach_word)
{
	/* mach_word is 4 on 32-bit machines, 8 on 64, etc */
	EiC_parseString(":reset here");
	sys_load_script(ext_comp->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("int mach_word @ %ld;", (long) &mach_word);
	EiC_parseString("add_data_types(mach_word);");

	EiCp_ResetStart();
	return (1);
}
int genargs_stub(struct function *f)
{
	/* mach_word is 4 on 32-bit machines, 8 on 64, etc */
	EiC_parseString(":reset here");
	sys_load_script(ext_comp->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("struct function * unsafe f @ %ld;", (long) f);
	EiC_parseString("comp_func_genargs(f);");

	EiCp_ResetStart();
	return (1);
}
int func_effect_stub(struct function *f)
{
	/* mach_word is 4 on 32-bit machines, 8 on 64, etc */
	EiC_parseString(":reset here");
	sys_load_script(ext_comp->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("struct function * unsafe f @ %ld;", (long) f);
	EiC_parseString("comp_func_effect(f);");

	EiCp_ResetStart();
	return (1);
}
int makestr_stub(char *buf, int len, int *pos, char *text, int text_len,
		 int type)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_comp->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("char * unsafe buf @ %ld;", (long) buf);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("int * unsafe pos @ %ld;", (long) pos);
	EiC_parseString("char * unsafe text @ %ld;", (long) text);
	EiC_parseString("int text_len @ %ld;", (long) &text_len);
	EiC_parseString("int type @ %ld;", (long) &type);
	EiC_parseString("makestr(buf, len, pos, text, text_len, type);");

	EiCp_ResetStart();
	return (1);
}

/* LANG stubs */
void lang_init_stub(void *param)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_lang->ext.filename);
	EiC_parseString("void * unsafe lang_init_param @ %ld;", (long) param);
	EiC_parseString("lang_init(lang_init_param);");
	EiCp_ResetStart();
	return;
}
void lang_cleanup_stub(void)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_lang->ext.filename);
	EiC_parseString("lang_cleanup( NULL);");
	EiCp_ResetStart();
	return;
}
int sprint_proto_stub(char *buf, int len, int func_id)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_lang->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("char * unsafe buf @ %ld;", (long) buf);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("int func_id @ %ld;", (long) &func_id);
	EiC_parseString("sprint_proto(buf, len, func_id);");

	EiCp_ResetStart();
	return (1);
}
int sprint_func_stub(char *buf, int len, int func_id)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_lang->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("char * unsafe buf @ %ld;", (long) buf);
	EiC_parseString("int len @ %ld;", (long) &len);
	EiC_parseString("int func_id @ %ld;", (long) &func_id);
	EiC_parseString("sprint_func(buf, len, func_id);");

	EiCp_ResetStart();
	return (1);
}
int gen_final_stub(int func_id)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_lang->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("int func_id @ %ld;", (long) &func_id);
	EiC_parseString("gen_final(func_id);");

	EiCp_ResetStart();
	return (1);
}
int gen_hllfile_stub(FILE *f)
{
	EiC_parseString(":reset here");
	sys_load_script(ext_lang->ext.filename);
	DEBUG_PrintMsg("executing %s\n", ext_lang->ext.filename);

	EiC_parseString("FILE * unsafe f @ %ld;", (long) f);
	EiC_parseString("gen_file(f);");

	EiCp_ResetStart();
	return (1);
}

/* safety wrappers -- called by routines outside of this file */
int ext_plugin_exec( char *plugin, char *name, void *param ) {
	char buf[64];
	int (*plugin_fn)(void *);
	struct EXTENSION *e;

	if ( ! plugin )	return(0);
	ext_plugin_change( plugin );
	e = &ext_plugin->ext;

	if ( (e->flags & EXT_METHOD_MASK) == EXT_SO ) {
		plugin_fn = dlsym(e->lib, name);
		if ( plugin_fn ){
			return( (*plugin_fn)(param) );
		}
	} else {
		EiC_parseString(":reset here");
		sys_load_script(e->filename);
		DEBUG_PrintMsg("executing %s\n", e->filename);

		EiC_parseString("void * unsafe param @ %ld;", (long) param);
		snprintf( buf, 64, "%s(param);", name );
		EiC_parseString(buf);
		EiCp_ResetStart();
	}
	return(1);
}

int ext_plugin_main(char *plugin, void *param)
{
	/* note: this one does all pre/post engines for plug-in */
	int rv = 0;

	if ( ! plugin )	return(0);
	ext_plugin_change( plugin );

	if (ext_plugin->fn_main) {
		do_engine_stage(ext_plugin->name, "pre", (int) param);
		rv = (*ext_plugin->fn_main) (param);
		do_engine_stage(ext_plugin->name, "post", (int) param);
	}
	return (rv);
}

int ext_engine_main(void *param)
{
	if (ext_engine->fn_main)
		return ((*ext_engine->fn_main) (param));
	return (0);
}

int ext_disasm_addr(char *buf, char tbl, struct code *c, long rva)
{
	if (ext_arch->fn_disasm_addr)
		return ((*ext_arch->fn_disasm_addr) (buf, tbl, c, rva));
	return (0);
}

int ext_disasm_invariant(unsigned char *in, int in_len, 
				 struct ARCH_INVARIANT *inv)
{
	if (ext_arch->fn_disasm_inv)
		return ((*ext_arch->fn_disasm_inv) (in, in_len, inv));
	return (0);
}

int ext_code_pat(int rva, int type)
{
	if (ext_arch->fn_code_pat)
		return ((*ext_arch->fn_code_pat) (rva, type));
	return (0);
}

int ext_gen_effect(struct code *c, struct code_effect *e)
{
	if (ext_arch->fn_gen_effect)
		return ((*ext_arch->fn_gen_effect) (c, e));
	return (0);
}

int ext_gen_int(struct function *f)
{
	if (ext_arch->fn_gen_int)
		return ((*ext_arch->fn_gen_int) (f));
	return (0);
}

int ext_add_data_types(int mach_word)
{
	if (ext_comp->fn_datatype)
		return ((*ext_comp->fn_datatype) (mach_word));
	return (0);
}

int ext_func_genargs( struct function *f ) {
	if (ext_comp->fn_genargs)
		return ((*ext_comp->fn_genargs) (f));
	return (0);
}

int ext_func_geneffect( struct function *f ) {
	if (ext_comp->fn_effect)
		return ((*ext_comp->fn_effect) (f));
	return (0);
}

int ext_findstr(char *buf, int len, int *pos, char *text, int text_len,
		int type)
{
	if (ext_comp->fn_findstr)
		return ((*ext_comp->fn_findstr)
			(buf, len, pos, text, text_len, type));
	return (0);
}

int ext_makestr(char *buf, char *text, int text_len, int type)
{
	if (ext_comp->fn_makestr)
		return ((*ext_comp->fn_makestr) (buf, text, text_len, type));
	return (0);
}

int ext_sysref_name( struct sysref *s, char *buf, int len)
{
	if (ext_os->fn_sysref_name)
		return ((*ext_os->fn_sysref_name) (s, buf, len));
	return (0);
}

int ext_sysref_gen( struct code *c, struct sysref *s)
{
	if (ext_os->fn_sysref_gen)
		return ((*ext_os->fn_sysref_gen) (c, s));
	return (0);
}

int ext_sprint_hll_proto(char *buf, int len, int func_id)
{
	if (ext_lang->fn_sprint_proto)
		return ((*ext_lang->fn_sprint_proto) (buf, len, func_id));
	return (0);
}

int ext_sprint_hll_func(char *buf, int len, int func_id)
{
	if (ext_lang->fn_sprint_func)
		return ((*ext_lang->fn_sprint_func) (buf, len, func_id));
	return (0);
}

int ext_gen_final(int func_id)
{
	if (ext_lang->fn_gen_final)
		return ((*ext_lang->fn_gen_final) (func_id));
	return (0);
}

int ext_gen_hll_file(FILE *f)
{
	if (ext_lang->fn_gen_file)
		return ((*ext_lang->fn_gen_file) (f));
	return (0);
}
int ext_read_header(void)
{
	if (ext_format->fn_read_header) {
		(*ext_format->fn_read_header) ();
		return (1);
	}
	return (0);
}
int ext_sprint_code(long rva, char *line, int len, int output)
{
	if (ext_asm->fn_sprint_code)
		return ((*ext_asm->fn_sprint_code) (rva, line, len, output));
	return (0);
}
int ext_sprint_addrexp(char *str, int len, char *scale, char *index,
		       char *base, char *disp, int sign)
{
	if (ext_asm->fn_sprint_addrexp)
		return ((*ext_asm->fn_sprint_addrexp)
			(str, len, scale, index, base, disp, sign));
	return (0);
}
int ext_sprint_asm_func_start(char *str, int len, int func_id)
{
	if (ext_asm->fn_sprint_func_s)
		return ((*ext_asm->fn_sprint_func_s) (str, len, func_id));
	return (0);
}
int ext_sprint_asm_func_end(char *str, int len, int func_id)
{
	if (ext_asm->fn_sprint_func_e)
		return ((*ext_asm->fn_sprint_func_e) (str, len, func_id));
	return (0);
}
int ext_sprint_asm_struct(char *str, int len, long rva)
{
	if (ext_asm->fn_sprint_struct)
		return ((*ext_asm->fn_sprint_struct) (str, len, rva));
	return (0);
}
int ext_sprint_sec_start(char *str, int len, char *name)
{
	if (ext_asm->fn_sprint_sec_s)
		return ((*ext_asm->fn_sprint_sec_s) (str, len, name));
	return (0);
}
int ext_sprint_sec_end(char *str, int len, char *name)
{
	if (ext_asm->fn_sprint_sec_e)
		return ((*ext_asm->fn_sprint_sec_e) (str, len, name));
	return (0);
}

int ext_gen_asm_file(FILE *f)
{
	if (ext_asm->fn_gen_file)
		return ((*ext_asm->fn_gen_file)(f));
	return (0);
}

int do_engine(char *plugin, char *prefix, int num, int options)
{
	char tmp[PATH_MAX];
	struct stat statbuf;

	sprintf(tmp, "%s/engines/%s/%s.%d.bc", disasm_env.share, plugin, prefix,
		num);
	if (stat(tmp, &statbuf) || !load_extension(EXT_ENGINE, tmp, ext_engine))
		return (sys_set_lasterr(4593));
	/* engines have no 'init' and thus are never passed ext_engine */
	return (ext_engine_main((void *)options));
}

int get_engine_num(char *str)
{
	int x;
	char *end, tmp[16] = { 0 };

	str = strrchr(str, '/');
	end = strstr(str, ".bc") - 1;
	x = end - str;
	while (x && str[x] != '.')
		x--;
	memcpy(tmp, &str[x + 1], end - &str[x]);
	return (atoi(tmp));
}

int do_engine_stage(char *plugin, char *stage, int options)
{
	char tmp[4096];
	glob_t globbuf;
	int x, num;

	sprintf(tmp, "%s/engines/%s/%s.*.bc", disasm_env.share, plugin, stage);
	glob(tmp, 0, NULL, &globbuf);
	for (x = 0; x < globbuf.gl_pathc; x++) {
		num = get_engine_num(globbuf.gl_pathv[x]);
		//if (num > 0 && num < 4)       /* hack to bypass EiC's brokenness */
		if (num > 0)	/* hack to bypass EiC's brokenness */
			do_engine(plugin, stage, num, options);
	}
	globfree(&globbuf);
	return (1);
}
