#include <cli/bastard_main.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <config.h>
#include <extension.h>

extern int binfromstdin, runbatch;	/* from bastard.c, used in QUIT */
extern FILE *batch_file;

/* --------------------------------------------------------------------- */

void cmdSET(int argc, char **argv, struct COMMAND *c)
{
	/* set internal ENV variables */
	char *opt, msg[64];
	int rv;

	if (!argc || !argv[0])
		opt = NULL;
	else
		opt = argv[0];

	if (env_set_option(opt) > 0 && (rv = env_get_option(opt)) > -1) {
		if (rv > 0)
			sprintf(msg, "%s = ON\n", opt);
		else
			sprintf(msg, "%s = OFF\n", opt);
		sys_msg(msg);
	}
	return;
}

void cmdSHOW(int argc, char **argv, struct COMMAND *c)
{
	/* show internal ENV variables */
	char *opt, msg[64];
	int rv;

	if (!argc || !argv[0])
		opt = NULL;
	else
		opt = argv[0];

	if ((rv = env_get_option(opt)) > -1) {
		if (rv > 0)
			sprintf(msg, "%s = ON\n", opt);
		else
			sprintf(msg, "%s = OFF\n", opt);
		sys_msg(msg);
	}
	return;
}

void cmdDB(int argc, char **argv, struct COMMAND *c)
{
	ProcessCommand("DB ?");
	return;
}

void cmdLOAD(int argc, char **argv, struct COMMAND *c)
{
	/* generic file load/disassembler. Assumes a.out, ELF, i386 */
	struct DISASM_ENV *disasm_env = env_get_env();
	struct DISASM_TGT *target = env_get_target();
	char cmd_buf[PATH_MAX], db_buf[PATH_MAX] = {0};
	char buf[PATH_MAX];
	unsigned char y;
	int err = 0;

	/* wrapper for multiple API routines */
	if (disasm_env->flags & DB_LOADED)
		target_close_db();	//duplicated in db_load but nec.

	if ( argc && argv[0][0] ) { 
		strncpy( buf, argv[0], PATH_MAX );
	} else if ( target->info.name && target->info.path ) {
		snprintf( buf, PATH_MAX, "%s/%s", target->info.path, 
				target->info.name );
	} else  {
		sys_msg("Please specify a target name\n");
		return;
	}

	if (! (target->status & DISASM_TGT_LOADED) && ! target_load(buf)) {
		err = sys_get_lasterr();
		/* Note that dbname has been set already by target_load() */
		if ( err == 4501 ) {
			sprintf(db_buf, "%s/%s", disasm_env->dbpath, 
					target->info.dbname);
		} else if ( err == 4502 ) {
			sprintf(db_buf, "%s/.%s", disasm_env->dbpath, 
					target->info.dbname);
		}

		/* first off, interact with the user if we have to */
		if ( env_get_opt_flag(ANNOY_USER) && err == 4501 ) {
			sys_msg( "Found previous .bdb ... use it? (y/n)\n");
			y = getc(stdin);
			getc(stdin);	/* strip CR */
			err = 0;
			if ( y == 'Y' || y == 'y' ) {
				if (! target_load_bdb( db_buf ) )
					err = sys_get_lasterr();
			} else {
				/* move and load target */
				sprintf(cmd_buf, "%s/%s.old", disasm_env->dbpath, 
						target->info.dbname);
				rename( db_buf, cmd_buf );
				if ( ! target_load(buf) )
					err = sys_get_lasterr();
			}
		} else if ( env_get_opt_flag(ANNOY_USER) && err == 4502 ) {
			sys_msg( "Found .bdb dir from previous load ... recover (y/n)?\n");
			y = getc(stdin);
			getc(stdin);
			err = 0;
			if ( y == 'Y' || y == 'y' ) {
				if (! target_load_bdb( db_buf ) )
					err = sys_get_lasterr();
			} else {
				/* delete and load target */
				sprintf(cmd_buf, "rm -rf %s", db_buf);
				pclose( popen(buf, "r") );
				if ( ! target_load(buf) )
					err = sys_get_lasterr();
			}
		} else if ( db_buf[0] ) {
			/* cool, we can just print an error message */
			sys_msg( db_buf );
			sys_msg( " already exists\n" );
		}

		if ( err ) {
			DEBUG_Print("target_load Failed!\n");
			sys_print_errmsg(err);
			return;
		}
	}

	/* target has now been loaded with default extensions */
	if (argc > 1 && argv[1][0]) {
		strncpy(target->format.name, argv[1], 15);
		target->status &= ~DISASM_TGT_FORMAT_SET;
	}
	if (argc > 2 && argv[2][0]) {
		strncpy(target->arch.name, argv[2], 31);
		target->status &= ~DISASM_TGT_ARCH_SET;
	}
	if (argc > 3 && argv[2][0]) {
		strncpy(target->os.name, argv[2], 31);
		target->status &= ~DISASM_TGT_OS_SET;
	}
	if (argc > 4 && argv[3][0]) {
		strncpy(target->assembler.name, argv[3], 31);
		target->status &= ~DISASM_TGT_ASM_SET;
	}
	if (argc > 5 && argv[2][0]) {
		strncpy(target->comp.name, argv[2], 31);
		target->status &= ~DISASM_TGT_COMP_SET;
	}
	if (argc > 6 && argv[4][0]) {
		strncpy(target->lang.name, argv[4], 31);
		target->status &= ~DISASM_TGT_LANG_SET;
	}

	if (! (target->status & DISASM_TGT_FORMAT_SET) &&
	    !target_set_format(target->format.name, target->format.options)) {
		DEBUG_Print("target_set_format Failed!\n");
		err = sys_get_lasterr();
	} else if (! (target->status & DISASM_TGT_ARCH_SET) &&
		   !target_set_arch(target->arch.name, target->arch.options)) {
		DEBUG_Print("target_set_arch Failed!\n");
		err = sys_get_lasterr();
	} else if (! (target->status & DISASM_TGT_OS_SET) &&
		   !target_set_os(target->os.name, target->os.options)) {
		DEBUG_Print("SetTargetOS Failed!\n");
		err = sys_get_lasterr();
	} else if (! (target->status & DISASM_TGT_ASM_SET) &&
		   !target_set_asm(target->assembler.name, 
			   			target->assembler.options)) {
		DEBUG_Print("SetTargetAsm Failed!\n");
		err = sys_get_lasterr();
	} else if (! (target->status & DISASM_TGT_COMP_SET) &&
		   !target_set_comp(target->comp.name, target->comp.options)) {
		DEBUG_Print("SetTargetComp Failed!\n");
		err = sys_get_lasterr();
	} else if (! (target->status & DISASM_TGT_LANG_SET) &&
		   !target_set_lang(target->lang.name, target->lang.options)) {
		DEBUG_Print("SetTargetLang Failed!\n");
		err = sys_get_lasterr();
	} else if (! (target->status & DISASM_TGT_DISASM) &&
		   !disasm_target("full", NULL)) {
		DEBUG_Print("disasm_target Failed!\n");
		err = sys_get_lasterr();
	}

	if (err)
		sys_print_errmsg(err);

	return;
}

void cmdEXEC(int argc, char **argv, struct COMMAND *c)
{
	int x, len;
	char cline[256] = { 0 };
	struct DISASM_ENV *disasm_env = env_get_env();

	if (!(disasm_env->flags & DISABLE_EXEC)) {
		if (argc && argv[0]) {
			sprintf(cline, "/bin/sh -c");
			for (x = 0; x < argc; x++) {
				len = 256 - strlen(cline);
				len = (len > 0) ? len : 0;
				strncat(cline, " ", len);
				strncat(cline, argv[x], --len);

			}
		} else {
			strcpy(cline, "/bin/sh -ri");
		}
		system(cline);
	}
	return;
}

void cmdDEBUG(int argc, char **argv, struct COMMAND *c)
{
	/* debug the target executable */
	/* maps to API routine */
	sys_debug();
	return;
}

void cmdQUIT(int argc, char **argv, struct COMMAND *c)
{
	/* leave the app */
	int x;
	FILE *f;
	char buf[PATH_MAX];
	HIST_ENTRY *hist;

	struct DISASM_ENV *disasm_env = env_get_env();
	struct DISASM_PREFS *disasm_prefs = env_get_prefs();
	struct DISASM_TGT *target = env_get_target();
	x = disasm_prefs->num_hist;

	sprintf(buf, "%s/.bastard/cmd_history", getenv("HOME"));
	if (f = fopen(buf, "w")) {
		while (x >= 0 && (hist = previous_history())) {
			if (hist->line[0] != '\n')
				fprintf(f, "%s\n", hist->line);
			x--;
		}
	}
	/* get history for saving in config db */
	/*while( x >= 0 && ( hist = previous_history() ) ){
	   AddHistCmd(x, hist->line);
	   x--;
	   } */
	sys_quit();
	env_set_flag(QUIT_NOW);
	/* why is this neceaary ?? */
	disasm_env->flags |= QUIT_NOW;

	return;
}

void cmdHEADER(int argc, char **argv, struct COMMAND *c)
{
	printf("%s\n", target_header());
	return;
}

void cmdSECTION(int argc, char **argv, struct COMMAND *c)
{
	char *tmp, name[32], secStr[128];
	struct section sec;
	struct address addr;
	void *it;
	int cont;

	if (!argc || !argv[0]) {
		sys_msg
		    ("Specify a section name when using 'SECTION'. Names:\n");
		cont = bdb_index_first(SECTION_NAME, &sec);
		while (cont) {
			sprintf(secStr, "%s\trva: %08X\tsize: %8X\n", sec.name,
				sec.rva, sec.size);
			sys_msg(secStr);
			cont = bdb_index_next(SECTION_NAME, &sec);
		}
	} else {
		if (!bdb_index_find(SECTION_NAME, argv[0], &sec))
			return;
		if (!(cont = bdb_index_find(ADDRESS_RVA, &sec.rva, &addr)))
			return;
		while (cont && addr.rva < (sec.rva + sec.size)) {
			addr_print(addr.rva);
			bdb_index_next(ADDRESS_RVA, &addr);
		}
	}
	return;
}

void cmdRANGE(int argc, char **argv, struct COMMAND *c)
{
	long start_rva, end_rva;
	char *s1 = NULL, *s2 = NULL;
	struct address addr;
	int cont, *state;

	if (argc < 2) {
		sys_msg("Specify a starting and ending address!\n");
		return;
	}
	/* get starting and ending address */
	start_rva = strtol(argv[0], NULL, 16);
	end_rva = strtol(argv[1], NULL, 16);
	if ((s1 && s1[0]) || (s2 && s2[0])) {
		sys_msg("Specify starting and ending addresses in hex!\n");
		return;
	}
	/* find closest addr (start_addr) */
	cont = bdb_find_closest_next(ADDRESS_RVA, &start_rva, &addr);

	/* while (addr< end_addr) addr = get_addr_from_db() */
	while (cont && addr.rva <= end_rva) {
		state = db_save_state();
		addr_print(addr.rva);
		db_restore_state(state);
		cont = bdb_index_next(ADDRESS_RVA, &addr);
	}

	return;
}

void cmdBDB(int argc, char **argv, struct COMMAND *c)
{
	if (!argc || !argv[0][0]) {
		sys_msg("Please specify a .bdb file to load!");
		return;
	}
	target_load_bdb(argv[0]);

	return;
}

void cmdINT(int argc, char **argv, struct COMMAND *c)
{
	int cont, block, *state;
	struct function f;
	struct int_code i;
	char buf[256] = {0};
	FILE *fd = NULL;
	int oldSTDOUT;

	if (argc && argv[0][0]) {
		oldSTDOUT = dup(STDOUT_FILENO);
		fd = freopen(argv[0], "w", stdout);
	}
	/* foreach function */
	cont = bdb_index_first(FUNCTION_ID, &f);
	while (cont) {
		state = db_save_state();
		/* print out int_code lines */
		//cont = bdb_index_find( FN_ORDER, &f.id, &i );
		cont = bdb_index_first( FN_ORDER, &i );
		while ( cont  ) {
			if ( f.id == i.func ) {
				if ( intcode_sprint( &i, buf, 256 ) )
					printf("%s\n",buf);
			} 
			cont = bdb_index_next(FN_ORDER, &i);
		}
		printf("\n\n");
		db_restore_state(state);
		cont = bdb_index_next(FUNCTION_ID, &f);
	}
	if (fd) {
		fclose(fd);
		dup2(oldSTDOUT, STDOUT_FILENO);
	}
	return;
}

void cmdDISASM(int argc, char **argv, struct COMMAND *c)
{
	struct address addr;
	struct section sec;
	int cont, *state;
	FILE *fd = NULL;
	int oldSTDOUT;
	char *filename = NULL;

	if (argc && argv[0][0]) {
		filename = argv[1];
	}
	target_output_as( "lst", filename, 0 );
	return;
}

void cmdDUMP(int argc, char **argv, struct COMMAND *cmd)
{
	int x, y;
	unsigned char hex[48 + 4], ascii[16 + 4], c;
	struct DISASM_TGT *target = env_get_target();

	for (x = 0; x < target->info.size; x += y) {
		memset(hex, 0, 52);
		memset(ascii, 0, 20);
		for (y = 0; x + y < target->info.size && y < 16; y++) {
			c = ((char *) target->image)[x + y];
			sprintf(hex, "%s%02X ", hex, c);
			if (isprint(c))
				ascii[y] = c;
			else
				ascii[y] = '.';
		}
		/* pad last line if < 16 bytes */
		if (y < 15)
			for (; y < 16; y++)
				sprintf(hex, "%s   ", hex);
		printf("%08X : %s\t%s\n", x, hex, ascii);
	}
}

void cmdSTRINGS(int argc, char **argv, struct COMMAND *c)
{
	str_print();
	return;
}

void cmdViewDocFile(char *filename)
{
	char cline[PATH_MAX];
	struct DISASM_ENV *disasm_env = env_get_env();
	struct DISASM_PREFS *disasm_prefs = env_get_prefs();

	snprintf(cline, PATH_MAX, "/bin/sh -c \'%s %s/doc/%s\'",
		 disasm_prefs->pager, disasm_env->share, filename);

	system(cline);
	return;
}
void cmdMAN(int argc, char **argv, struct COMMAND *c)
{
	cmdViewDocFile("bastard.txt");
	return;
}
void cmdAPI(int argc, char **argv, struct COMMAND *c)
{
	cmdViewDocFile("API.txt");
	return;
}
void cmdHOWTO(int argc, char **argv, struct COMMAND *c)
{
	cmdViewDocFile("Disasm-HOWTO.txt");
	return;
}


void cmdMACRO(int argc, char **argv, struct COMMAND *c)
{
	/* write a BC script */
	/* maps to API routine */
	struct DISASM_ENV *disasm_env = env_get_env();
	char *line, *script = NULL;
	int cont = 1;

	while (cont) {
		line = (char *) readline(disasm_env->p3);
		if (!istrncmp(line, "ENDMACRO", 8))
			cont = 0;
		else {
			script = realloc(script, strlen(line));
			strcat(script, line);
		}
	}
	if (strlen(script) < 2048) {
		/* macro_new( name, script); */
	} else {
		sys_msg("Macro is too large to save in database.\n"
			"Enter a filename for the macro:");
		/* ** name will be .bdb/macros/$name.sc */
		/* gets, fopen, fwrite, fclose */
		/* id = macro_new( name, script ); */
		/* SetMacroType( id, MACOR_FILE ); */
	}

	return;
}

void cmdENDMACRO(int argc, char **argv, struct COMMAND *c)
{
	sys_msg("ENDMACRO is only valid in the context of a MACRO block!\n");
	return;
}

void cmdRECORD(int argc, char **argv, struct COMMAND *c)
{
	/* record commandline actions ... may be useless */
	/* maps to API routine */
	sys_msg("Not yet implemented\n");
	return;
}

void cmdSTOP(int argc, char **argv, struct COMMAND *c)
{
	sys_msg("Not yet implemented\n");
	return;
}

void cmdRUN(int argc, char **argv, struct COMMAND *c)
{
	/* Run Macro */
	if (argc && argv[0][0])
		script_file_exec(argv[0]);
	return;
}

void cmdBRACE(int argc, char **argv, struct COMMAND *c)
{
	char *line, path[512];
	int x, cont = 1;
	FILE *f;
	struct DISASM_ENV *disasm_env = env_get_env();

	snprintf(path, 512, "%s/.tmp_macro", disasm_env->dbpath);
	f = fopen(path, "w");
	if (!f) {
		sys_print_errmsg(3020);
		return;
	}
	fputs("#include <string.h>\n", f);
	fputs("#include <unistd.h>\n", f);
	fputs("#include <stdlib.h>\n", f);
	fputs("int main(void) {\n", f);
	
	if ( runbatch ) line = calloc( PATH_MAX, 1 );
	if (! line) {
		fclose(f);
		return;
	}
	while (cont) {
		if ( runbatch ) {
			fgets( line, PATH_MAX -1, batch_file);
		} else {
			line = (char *) readline(disasm_env->p3);
		}
		if (!strncmp(line, "}}", 2))
			cont = 0;
		else {
			fprintf(f, "%s\n", line);
		}
		if (line && ! runbatch)
			free(line);
	}
	if (runbatch) free(line);

	fputs("return(0);\n}", f);
	fclose(f);

	sys_exec_script(path, 0, NULL);
	return;
}

void cmdOUTPUT(int argc, char **argv, struct COMMAND *c)
{
	char *filename = NULL;

	/* Call Output Plugin */
	if (argc && argv[0][0]) {
		if ( argc > 1 && argv[1][0] ) {
			/* no arg means STDOUT */
			filename = argv[1];
		}
		target_output_as( argv[0], filename, 0 );
	}
	return;
}

void cmdPLUGIN(int argc, char **argv, struct COMMAND *c)
{
	unsigned long options = 0;
	char *ptr, *funcname = NULL;
	void * plugin;
	/* for now, we ignore 'arg' */

	/* PLUGIN name [function] [arg] */
	/* Call Output Plugin */
	if (argc && argv[0][0]) {
		plugin = plugin_load( argv[0], 0 );
		if (! plugin ) {
			fprintf( stderr, "Unable to load plugin argv[0]\n");
			return;
		}
		if ( argc > 1 && argv[1][0] ) {
			if ( argc > 2 ) {
				/* an option was passed ! */
				options = strtoul( argv[2], &ptr, 0 );
				if ( *ptr ) {
					fprintf( stderr, "Error in %s: PLUGIN args must "
						   "be passed in numeric format\n", argv[2] );
					return;
				}
			}
			funcname = argv[1];
			plugin_exec( plugin, funcname, (void *)options );
		} else {
			plugin_exec_main( plugin, NULL );
		}
		plugin_unload( plugin );
	}
	return;
}

void cmdSIGGEN(int argc, char **argv, struct COMMAND *c){
	FILE *f;
	void *plugin;
	struct name n;
	char buf[PATH_MAX], *entry_sym = NULL;
	struct DISASM_TGT *target = env_get_target();
	
	if (argc && argv[0][0]) {
		strncpy( buf, argv[0], PATH_MAX );
	} else {
		snprintf( buf, PATH_MAX, "%s/plugins/signatures/%s.sig", 
				env_get_share(), target->info.name );
	}
	if ( argc > 2 && argv[1][0] ) {
		entry_sym = argv[1];
	} else {
		/* if _start */
		if ( bdb_index_find(NAME_TEXT, "_start", &n) ) {
			entry_sym = "_start";
		}
	}

	f = fopen( buf, "w+" );
	if ( ! f ) {
		fprintf( stderr, "Cannot write .sig file (%s): %s\n", buf, 
				strerror(errno) );
		return;
	}

	plugin = plugin_load( "signatures", 0 );
	if ( plugin ) {
		/* um, entry sym :) */
		if ( plugin_exec( plugin, "sign_gen_target", entry_sym ) &&
		     plugin_exec( plugin, "sign_write_file", f )  )   {
			sys_msg( "Signature file written to ");
			sys_msg( buf );
			sys_msg( "\n");
		}
		plugin_unload( plugin );
	} else {
		fprintf( stderr, "Cannot load plugin: %s\n", sys_lasterr_str() );	
	}

	fclose( f );
	
	return;
}

void cmdSIGLOAD(int argc, char **argv, struct COMMAND *c){
	char buf[PATH_MAX];
	struct stat s;
	FILE *f;
	void *plugin;
	struct DISASM_TGT *target = env_get_target();

	if (argc && argv[0][0]) {
		strncpy( buf, argv[0], PATH_MAX );
		if ( stat( buf, &s ) ) {
			/* not explicit path; try sigs dir */
			snprintf( buf, PATH_MAX, "%s/plugins/signatures/%s", 
					env_get_share(), argv[0] );
			if ( stat( buf, &s ) ) {
				fprintf(stderr, "Cannot load .sig file (%s): %s\n", 
					buf, strerror(errno));
				return;
			}
		}
		/* ok, at this point we can load the .sig file */
		f = fopen( buf, "r" );
		if ( ! f ) {
			fprintf( stderr, "Cannot write .sig file (%s): %s\n", buf, 
				strerror(errno) );
			return;
		}

		plugin = plugin_load( "signatures", 0 );
		if ( plugin ) {
			/* um, entry sym??? :) */
			plugin_exec( plugin, "sign_load_file", f );
			/* note: do not unload libsigs at this point! */
		} else {
			fprintf( stderr, "Cannot load plugin: %s\n", 
					sys_lasterr_str() );	
		}

		fclose( f );
	} else {
		/* list all signature files */
		fprintf(stderr, "Please specify a signature file to load!\n");
		snprintf( buf, PATH_MAX, "%s/plugins/signatures", 
				env_get_share() );
	}
	return;
}

static void sig_func_cb(void *item, void *plugin){
	void *state;
      struct function *f = item;

	if ( ! plugin || ! f )	return;

	state = db_save_state();
	plugin_exec( plugin, "sign_match_func", f );
	db_restore_state( state );
	return;
} 

void cmdSIGMATCH(int argc, char **argv, struct COMMAND *c){
	void * plugin;
	struct section sec = {0};
	struct DISASM_TGT *target = env_get_target();
	
	plugin = plugin_load( "signatures", 0 );

	if ( ! plugin ) {
		fprintf( stderr, "Cannot load plugin: %s\n", sys_lasterr_str() );	
	}

	if (argc && ! strcmp( argv[0], "SEC" ) ) {
		if ( argc == 2 ) {

		/* do matches for a section */
		/* find section by name -- argv[1] */
		plugin_exec( plugin, "sign_match_sec", &sec );
		} else {
			fprintf( stderr, "Usage: sig match sec section-name\n");
		}
	} else {
		/* do matches for all functions */
		func_foreach( sig_func_cb, plugin );
	}

	return;
}

void cmdSIGID(int argc, char **argv, struct COMMAND *c){
	char buf[PATH_MAX];
	struct DISASM_TGT *target = env_get_target();
	if (argc && argv[0][0]) {
		strncpy( buf, argv[0], PATH_MAX );
	}
	printf("SIG IDENTIFY not yet implemented\n");
	return;
}
void cmdSIG(int argc, char **argv, struct COMMAND *c){
	ProcessCommand("SIG ?");
	return;
}

void cmdVERSION(int argc, char **argv, struct COMMAND *c){
	struct DISASM_TGT *target = env_get_target();
	struct DISASM_ENV *env = env_get_env();
	printf("B A S T A R D   v. %.2f\n", BASTARD_VERSION);
	printf("Extensions:\n");
	ext_fprint_versions( stdout );
	printf( "Base Dir: %s \n", env_get_base() );
	printf( "DB Output Path: %s\n", env->dbpath );
	printf("Pager: %s Editor: %s Debugger: %s\n", env_get_pager(), 
			env_get_editor(), env_get_debugger());
	if ( target->info.name[0] ) {
		printf("Target:\n");
		printf("\t%s %0.2f\n",target->info.name,target->info.version);
		if ( target->info.path[0] )
			printf("\tPath: %s\n",  target->info.path );
		if ( target->info.dbname[0] )
			printf("\tDB Name: %s\n", target->info.dbname );
		if ( target->info.comment[0] )
			printf("\tComment: %s\n", target->info.comment );
		if ( target->info.vendor[0] )
			printf("\tVendor: %s\n", target->info.vendor );
		if ( target->info.model[0] )
			printf("\tModel: %s\n", target->info.model );
	}
	return;
}

void cmdCOMMENT(int argc, char **argv, struct COMMAND *c){
	unsigned long addr = 0;
	struct comment cmt;
	struct function f;
	struct address a;
	struct name n;
	char *name, *text;

	if ( argc && argv[0][0] ) {
		name = argv[0];

		if ( name[0] == '0' && (name[1] == 'x' || name[1] == 'X') ){
			addr = strtoul( name, NULL, 16 );
		} 

		if ( ! addr ) {
			/* try to find 'name */
			if ( bdb_index_find( NAME_TEXT, name, &n ) ){
				addr = n.rva;
			} else {
				fprintf( stderr, "Invalid name \"%s\"\n", name );
				return;
			}
		}

		/* at this point, 'addr' is set */
		if ( argc > 1 ) {
			/* SET COMMENT */
			text = argv[1];
			if ( bdb_index_find( FUNCTION_RVA, &addr, &f ) ) {
				/* 'addr' is of a function */
				if ( f.comment &&
				     bdb_index_find( COMMENT_ID, &f.comment, &cmt ) ) {
					/* update comment for function */
					cmt.type = CMT_USER;
					strncpy( cmt.text, text, 255 );
					bdb_record_update( COMMENT_ID, &cmt.id, &cmt );
				} else {
					/* insert comment for function */
					cmt.type = CMT_USER;
					f.comment = comment_new( text, CMT_USER );
					bdb_record_update( FUNCTION_ID, &f.id, &f );
				}
			} else if ( bdb_index_find( ADDRESS_RVA, &addr, &a ) ) {
				if ( a.comment &&
				     bdb_index_find( COMMENT_ID, &a.comment, &cmt ) ) {
					/* update comment for address */
					cmt.type = CMT_USER;
					strncpy( cmt.text, text, 255 );
					bdb_record_update( COMMENT_ID, &cmt.id, &cmt );
				} else {
					/* insert comment for address */
					cmt.type = CMT_USER;
					a.comment = comment_new( text, CMT_USER );
					bdb_record_update( ADDRESS_RVA, &a.rva, &a );
				}
			} else {
				fprintf( stderr, "Invalid address %08X\n", addr );
			}
		} else {
			/* PRINT COMMENT */
			if ( bdb_index_find( FUNCTION_RVA, &addr, &f ) ) {
				if ( f.comment &&
				     bdb_index_find( COMMENT_ID, &f.comment, &cmt ) ) {
					printf("%X: \"%s\"\n", cmt.id, cmt.text);
				}
			} else if ( bdb_index_find( ADDRESS_RVA, &addr, &a ) ) {
				if ( a.comment &&
				     bdb_index_find( COMMENT_ID, &a.comment, &cmt ) ) {
					printf("%X: \"%s\"\n", cmt.id, cmt.text);
				}
			}
		}
	} else {
		printf("Specify a name ['loc_8048100'] or address ['0x8048100']\n");
	}

	return;
}

static void cmd_print_code_range( unsigned long start, unsigned long end ){
	struct code c = {0};
	void *state;
	int cont;

	cont = bdb_find_closest_prev( CODE_RVA, &start, &c );
	while ( cont && c.rva < end) {
		state = db_save_state();
		addr_print(c.rva);
		db_restore_state( state );
		cont = bdb_index_next( CODE_RVA, &c );
	}
	return;
}

static void cmd_print_data_range( unsigned long pa, int size ){
	int x, y;
	unsigned char hex[48 + 4], ascii[16 + 4], c;
	struct DISASM_TGT *target = env_get_target();

	if ( pa + size >= target->info.size ) {
		fprintf( stderr, "Error: invalid physical address %X\n", pa );
		return;
	}
	for (x = pa; x < target->info.size && x < (pa+size); x += y) {
		memset(hex, 0, 52);
		memset(ascii, 0, 20);
		for (y = 0; x + y < target->info.size && y < 16 
				&& x + y < (pa+size);               y++) {
			c = ((char *) target->image)[x + y];
			sprintf(hex, "%s%02X ", hex, c);
			if (isprint(c))
				ascii[y] = c;
			else
				ascii[y] = '.';
		}
		/* pad last line if < 16 bytes */
		if (y < 15)
			for (; y < 16; y++)
				sprintf(hex, "%s   ", hex);
		printf("%08X : %s\t%s\n", x, hex, ascii);
	}
	return;
}

void cmdNAME(int argc, char **argv, struct COMMAND *c){
	unsigned long addr = 0;
	char *name, *rename;
	struct address a;
	struct name n;
	int cont;

	if ( ! argc ) {
		/* PRINT */
		cont = bdb_index_first( NAME_TEXT, &n );
		while ( cont ) {
			printf( "\"%s\"\tType: %08X Rva: %08X\n",n.text, n.type, 
					n.rva );
			cont = bdb_index_next( NAME_TEXT, &n );
		}
	} else {
		name = argv[0];
		if ( argc > 1 && argv[1][0]) {
			/* RENAME */
			rename = argv[1];
			if ( name[0] == '0' && (name[1] == 'x' || name[1] == 'X') ){
				addr = strtoul( name, NULL, 16 );
			}
			if ( addr ) {
				/* try to rename 'address' */
				if ( bdb_index_find( NAME_RVA, &addr, &n ) ){
					strncpy( n.text, rename, 63 );
					n.type = (n.type & ~0x0F) | NAME_USER;
					bdb_record_update( NAME_RVA, &addr, &n );
				} else if ( bdb_index_find( ADDRESS_RVA, &addr, &a ) ) {
					/* make sure this is a real address */
					strncpy( n.text, rename, 63 );
					n.rva = addr;
					n.type = NAME_USER;
					bdb_record_insert(NAME, &n);
				} else {
					fprintf( stderr, "Invalid address %08X\n", addr );
				}
			} else {
				/* try to rename 'name */
				if ( bdb_index_find( NAME_TEXT, name, &n ) ){
					strncpy( n.text, rename, 63 );
					n.type = (n.type & ~0x0F) | NAME_USER;
					bdb_record_insert(NAME, &n);
				} else {
					fprintf( stderr, "Invalid name \"%s\"\n", name );
				}
			}	/* end if-addr-or-name */
				
		} else {
			/* DUMP */
			if ( bdb_index_find( NAME_TEXT, name, &n ) ){
				addr = func_get_end( n.rva );
				if ( addr ) {
					cmd_print_code_range( n.rva, addr );
				} else {
					if ( bdb_index_find( ADDRESS_RVA, &n.rva, &a )) {
						printf("%s:\n", n.text);
						cmd_print_data_range( a.pa, a.size );
					} else {
						fprintf(stderr, "Unknown address %s (%08X)\n", 
								n.text, n.rva );
					}
				}
			} else {
				fprintf( stderr, "Invalid name \"%s\"\n", name );
			}
		}	/* end if-rename-or-dump */
	}	/* end if-print-or-rename/dump */
	return;
}

void cmdENTRY(int argc, char **argv, struct COMMAND *c){
	unsigned long addr;
	struct DISASM_TGT *target = env_get_target();

	if ( target->info.entry == 0xFFFFFFFF ) {
		fprintf(stderr, "Error: target has no entry point!\n");
		return;
	}
	addr = func_get_end( target->info.entry );
	if ( ! addr ) 
		addr = target->info.entry + 128; /* enough for a screen */
	cmd_print_code_range( target->info.entry, addr );

	return;
}

static void cmd_db_findmnem( char *mnem ) {
	int cont;
	struct code_mnemonic cm = {0};
	struct code c;
	void *state;

	strncpy( cm.mnemonic, mnem, 16 );
	cont = bdb_index_find( CODE_DEST_OP, &cm, &c );
	while ( cont ) {
		if ( strcmp( c.mnemonic, mnem ) ) {
			cont = 0;
			continue;
		}
		/* print insn */
		state = db_save_state();
		addr_print(c.rva);
		db_restore_state( state );
		cont = bdb_index_next( CODE_MNEMONIC, &c );
	}
	return;
}

static void cmd_db_findop( unsigned long val, int type ) {
	int cont;
	struct code_dest_op cd = {0};
	struct code_src_op cs = {0};
	struct code c;
	void *state;

	/* if ! type */
	printf("Occurrences as Destination Operand:\n");
	cd.dest = (long) val;
	cont = bdb_find_closest_next( CODE_DEST_OP, &cd, &c );
	while ( cont ) {
		if ( c.dest != (long) val ) {
			cont = 0;
			continue;
		}
		if ( ( type && OP_TYPE(c.destType) == type) ||
		     ( !type && 
			   OP_TYPE(c.destType) != OP_EXPR &&
			   OP_TYPE(c.destType) != OP_REG     )       ){
			state = db_save_state();
			addr_print(c.rva);
			db_restore_state( state );
		}
		cont = bdb_index_next( CODE_DEST_OP, &c );
	}
	printf("Occurrences as Src Operand:\n");
	cs.src = val;
	cont = bdb_find_closest_next( CODE_SRC_OP, &cs, &c );
	while ( cont ) {
		if ( c.src != (long) val ) {
			cont = 0;
			continue;
		}
		if ( ( type && OP_TYPE(c.srcType) == type) ||
		     ( !type && 
			   OP_TYPE(c.destType) != OP_EXPR &&
			   OP_TYPE(c.destType) != OP_REG     )       ){
			state = db_save_state();
			addr_print(c.rva);
			db_restore_state( state );
		}
		cont = bdb_index_next( CODE_SRC_OP, &c );
	}
	return;
}

void cmdFINDINSN(int argc, char **argv, struct COMMAND *c){
	unsigned long val;
	if ( argc != 1 ) {
		fprintf(stderr, "Specify a mnemonic [e.g. int3]\n");
		return;
	}
	cmd_db_findmnem(argv[0]);
	return;
}

void cmdFINDOP(int argc, char **argv, struct COMMAND *c){
	unsigned long val;
	if ( argc != 1 ) {
		fprintf(stderr, "Specify a numeric operand [e.g. 16 or 0x1234]\n");
		return;
	}
	val = strtoul( argv[0], NULL, 0 );
	cmd_db_findop(val, 0);
	return;
}
void cmdFINDREG(int argc, char **argv, struct COMMAND *c){
	unsigned long val;
	if ( argc != 1 ) {
		fprintf(stderr, "Specify a register [e.g. eax]\n");
		return;
	}
	val = vm_get_reg_byname(argv[0]);
	cmd_db_findop(val, OP_REG);
	return;
}

static void cmd_db_findrefs( unsigned long rva ) {
	int cont;
	void *state;
	struct xref x;
	struct xref_to_rva xt = {0};

	xt.to_rva = rva;
	cont = bdb_find_closest_next( XREF_TO_RVA, &xt, &x );
	while (cont) {
		if ( xt.to_rva != rva ) {
			cont = 0;
			continue;
		}
		state = db_save_state();
		addr_print(xt.from_rva);
		db_restore_state( state );
		cont = bdb_index_next( XREF_TO_RVA, &x );
	}
	return;
}

void cmdFINDNAME(int argc, char **argv, struct COMMAND *c){
	unsigned long val;
	struct name n;

	if ( argc != 1 ) {
		fprintf(stderr, "Specify a name [e.g. _start]\n");
		return;
	}

	if ( bdb_index_find( NAME_TEXT, argv[0], &n ) ) {
		cmd_db_findrefs( n.rva );
	} else {
		fprintf( stderr, "Unknown name \"%s\"\n", argv[0] );
	}
	return;
}
void cmdFINDXREF(int argc, char **argv, struct COMMAND *c){

	if ( argc != 1 ) {
		fprintf(stderr, "Specify an address [e.g. 8048100]\n");
		return;
	}
	cmd_db_findrefs( strtoul( argv[0], NULL, 0 ) );
	return;
}

void cmdFINDBYTES(int argc, char **argv, struct COMMAND *c){
	int i, j, count = 0;
	unsigned char b;
	char *buf;
	struct DISASM_TGT *target = env_get_target();

	if ( argc != 1 ) {
		fprintf(stderr, "Specify a series of bytes [e.g. \"00 01 40 08\"]\n");
		return;
	}
	/* get a count of number of bytes */
	for ( i = 0; i < strlen(argv[0]); i++ ) {
		if ( argv[0][i] == '\'' || argv[0][i] == '\'' ){
			argv[0][i] = ' ';	/* remove quotes */
			continue;
		}
		if ( isalnum( argv[0][i] ) ) {
			count++;
			while ( i < strlen(argv[0]) && isalnum(argv[0][i]) ) 
				i++;
		}
	}
	buf = calloc( count, 1 );
	if ( ! buf ) {
		sys_set_lasterr(9050);
		return;
	}
	count = 0;

	for ( i = 0; i < strlen(argv[0]); i++ ) {
		if ( isalnum( argv[0][i] ) ) {
			b = (unsigned char) strtoul(&argv[0][i], NULL, 16);
			buf[count] = b;
			count++;
			while ( i < strlen(argv[0]) && isalnum(argv[0][i]) ) 
				i++;
		}
	}

	for ( i = 0; i < target->info.size; i++ ) {
		for ( j = 0; j < count && i + j < target->info.size && 
				 buf[j] == target->image[i+j];              j++ )
			;
		if ( j == count ) {
			addr_print(addr_rva(i+j));
			i+=j;		/* skip past found block */
			i--;		/* compensate for the auto-inc */
		}
	}
	free(buf);
	return;
}

void cmdFIND(int argc, char **argv, struct COMMAND *c){
	//fprintf(stderr, "Usage: FIND [HELP|INSN|OP|REG|XREF|NAME|BYTES] value\n");
	ProcessCommand("FIND ?");
	return;
}

void cmdPRELOAD(int argc, char **argv, struct COMMAND *c)
{
	/* load target but do not apply any extensions/plugins */
	struct DISASM_ENV *disasm_env = env_get_env();
	struct DISASM_TGT *target = env_get_target();
	char cmd_buf[PATH_MAX], db_buf[PATH_MAX];
	char buf[PATH_MAX];
	unsigned char y;
	int err = 0;

	/* wrapper for multiple API routines */
	if (disasm_env->flags & DB_LOADED)
		target_close_db();	//duplicated in db_load but nec.

	if ( argc && argv[0][0] ) { 
		strncpy( buf, argv[0], PATH_MAX );
	} else if ( target->info.name && target->info.path ) {
		snprintf( buf, PATH_MAX, "%s/%s", target->info.path, 
				target->info.name );
	} else  {
		sys_msg("Please specify a target name\n");
		return;
	}

	if ( !(target->status & DISASM_TGT_LOADED) && ! target_load(buf)) {
		err = sys_get_lasterr();
		/* Note that dbname has been set already by target_load() */
		if ( env_get_opt_flag(ANNOY_USER) && err == 4501 ) {
			sprintf(db_buf, "%s/%s", disasm_env->dbpath, target->info.dbname);
			sys_msg( "Found previous .bdb ... use it? (y/n)\n");
			y = getc(stdin);
			getc(stdin);	/* strip CR */
			err = 0;
			if ( y == 'Y' || y == 'y' ) {
				if (! target_load_bdb( db_buf ) )
					err = sys_get_lasterr();
			} else {
				/* move and load target */
				sprintf(cmd_buf, "%s/%s.old", disasm_env->dbpath, target->info.dbname);
				rename( db_buf, cmd_buf );
				if ( ! target_load(buf) )
					err = sys_get_lasterr();
			}
		} else if ( env_get_opt_flag(ANNOY_USER) && err == 4502 ) {
			sprintf(db_buf, "%s/.%s", disasm_env->dbpath, target->info.dbname);
			sys_msg( "Found .bdb dir from previous load ... recover (y/n)?\n");
			y = getc(stdin);
			getc(stdin);
			err = 0;
			if ( y == 'Y' || y == 'y' ) {
				if (! target_load_bdb( db_buf ) )
					err = sys_get_lasterr();
			} else {
				/* delete and load target */
				sprintf(cmd_buf, "rm -rf %s", db_buf);
				pclose( popen(buf, "r") );
				if ( ! target_load(buf) )
					err = sys_get_lasterr();
			}
		} 
		if ( err ) {
			DEBUG_Print("target_load Failed!\n");
			sys_print_errmsg(err);
			return;
		}
	}

	/* target has now been loaded with default extensions */

	return;
}

extern struct EXT__ARCH *ext_arch;
extern struct EXT__ASM *ext_asm;
extern struct EXT__COMP *ext_comp;
extern struct EXT__FORMAT *ext_format;
extern struct EXT__LANG *ext_lang;
extern struct EXT__OS *ext_os;

void cmdEXT(int argc, char **argv, struct COMMAND *c){
	unsigned long options = 0;
	struct EXTENSION *ext;
	char *ptr = NULL;
	int i;

	if ( argc && argv[0][0] ) { 
		for ( i = 0; i < strlen(argv[0]); i++ )
			argv[0][i] = toupper(argv[0][i]);

		/* select extension */
		if ( ! strcmp( argv[0], "ARCH" ) ) {
			ext = (struct EXTENSION *) &ext_arch->ext;
		} else if ( ! strcmp( argv[0], "ASM" ) ) {
			ext = (struct EXTENSION *) &ext_asm->ext;
		} else if ( ! strcmp( argv[0], "COMP" ) ) {
			ext = (struct EXTENSION *) &ext_comp->ext;
		} else if ( ! strcmp( argv[0], "FORMAT" ) ) {
			ext = (struct EXTENSION *) &ext_format->ext;
		} else if ( ! strcmp( argv[0], "LANG" ) ) {
			ext = (struct EXTENSION *) &ext_lang->ext;
		} else if ( ! strcmp( argv[0], "OS" ) ) {
			ext = (struct EXTENSION *) &ext_os->ext;
		} else {
			fprintf( stderr, "Unknown extension %s\n", argv[0] );
			return;
		}

		if ( argc > 1 ) {
			if ( argc > 2 ) {
				/* an option was passed ! */
				options = strtoul( argv[2], &ptr, 0 );
				if ( *ptr ) {
					fprintf( stderr, "Error in %s: EXT options must "
						   "be passed in numeric format\n", argv[2] );
					return;
				}
			}
			/* SET extension */
			if ( ext == &ext_arch->ext ) {
				target_set_arch( argv[1], options );
			} else if ( ext == &ext_asm->ext ) {
				target_set_asm( argv[1], options );
			} else if ( ext == &ext_comp->ext ) {
				target_set_comp( argv[1], options );
			} else if ( ext == &ext_format->ext ) {
				target_set_format( argv[1], options );
			} else if ( ext == &ext_lang->ext ) {
				target_set_lang( argv[1], options );
			} else if ( ext == &ext_os->ext ) {
				target_set_os( argv[1], options );
			}
		} else {
			/* PRINT extension */
			printf("Extension %s: %s\n", argv[0], ext->filename );
		}
	} else {
		printf("Extension ARCH : %s\n", ext_arch->ext.filename );
		printf("Extension ASM : %s\n", ext_asm->ext.filename );
		printf("Extension COMP : %s\n", ext_comp->ext.filename );
		printf("Extension FORMAT : %s\n", ext_format->ext.filename );
		printf("Extension LANG : %s\n", ext_lang->ext.filename );
		printf("Extension OS : %s\n", ext_os->ext.filename );
	}
	return;
}

