/*  Disassembly Database for BASTARD */
/*  
 *  SECTION     -- Distinct regions of code or data within the program
 *  ADDRESS     -- Distinct code or data units within the program
 *  CODE        -- An address object interpreted as a machine instruction
 *  NAME        -- Unique text strings associated with address objects
 *  COMMENT     -- Text strings associated with address objects
 *  IMPORT_ADDR -- External code references in the program
 *  EXPORT_ADDR -- Program Entry Points
 *  STRING      -- Data regions considered strings
 *  FUNCTION    -- Code regions defined as a function
 *  F_INLINE    -- Code regions associated with C macro/inline function
 *  STRUCTURE   -- Data regions associated with a defined structure
 *  CONSTANT    -- User-defined enumerations [from .h files]
 *  BC_MACRO    -- User-defined bc macros [char strings]
 */


database target {

	data file  "address.db"   contains address;
	key  file  "address.ix1"  contains address.rva;
	key  file  "address.ix2"  contains address.pa;
	data file  "section.db"   contains section;
	key  file  "section.ix1"  contains section.name;
	key  file  "section.ix2"  contains section.rva;
	data file  "xref.db"      contains xref;
	key  file  "xref.ix1"     contains xref.id;
	key  file  "xref.ix2"     contains xref.xref_from_rva;
	key  file  "xref.ix3"     contains xref.xref_to_rva;
	key  file  "xref.ix4"     contains xref.from_rva;
	key  file  "xref.ix5"     contains xref.to_rva;
	data file  "code.db"      contains code;
	key  file  "code.ix1"     contains code.rva;
	key  file  "code.ix2"     contains code.code_dest_op;
	key  file  "code.ix3"     contains code.code_src_op;
	key  file  "code.ix4"     contains code.code_mnemonic;
	key  file  "code.ix5"     contains code.code_function;
	data file  "addr_exp.db"  contains addr_exp;
	key  file  "addr_exp.ix1" contains addr_exp.id;
	data file  "code_effect.db"  contains code_effect;
	key  file  "code_effect.ix1" contains code_effect.id;
	key  file  "code_effect.ix2" contains code_effect.code_effect_rvaid;
	data file  "reg_state.db" contains reg_state;
	key  file  "reg_state.ix1" contains reg_state.id;
	key  file  "reg_state.ix2" contains reg_state.reg_state_by_ref;
	data file  "sysref.db" contains sysref;
	key  file  "sysref.ix1" contains sysref.id;
	key  file  "sysref.ix2" contains sysref.sysref_type_id;
	key  file  "sysref.ix3" contains sysref.sysref_from;
	data file  "int_code.db"  contains int_code;
	key  file  "int_code.ix1" contains int_code.id;
	key  file  "int_code.ix2" contains int_code.rva;
	key  file  "int_code.ix3" contains int_code.fn_order;
	data file  "fin_code.db"  contains fin_code;
	key  file  "fin_code.ix1" contains fin_code.id;
	key  file  "fin_code.ix2" contains fin_code.rva;
	data file  "export.db"    contains export_addr;
	key  file  "export.ix1"   contains export_addr.rva;
	data file  "import.db"    contains import_addr;
	key  file  "import.ix1"   contains import_addr.rva;
	key  file  "import.ix2"   contains import_addr.import_library;
	data file  "lib.db"    	contains lib;
	key  file  "lib.ix1"   	contains lib.id;
	key  file  "lib.ix2"   	contains lib.name;
	data file  "string.db"    contains string;
	key  file  "string.ix1"   contains string.rva;
	data file  "name.db"      contains name;
	key  file  "name.ix1"     contains name.id;
	key  file  "name.ix2"     contains name.rva;
	key  file  "name.ix3"     contains name.text;
	data file  "comment.db"   contains comment;
	key  file  "comment.ix1"  contains comment.id;
	data file  "function.db"   contains function;
	key  file  "function.ix1"  contains function.id;
	key  file  "function.ix2"  contains function.rva;
	data file  "func_param.db"   contains func_param;
	key  file  "func_param.ix1"  contains func_param.id;
	key  file  "func_param.ix2"  contains func_param.func_param_funcid;
	data file  "func_local.db"   contains func_local;
	key  file  "func_local.ix1"  contains func_local.id;
	key  file  "func_local.ix2"  contains func_local.func_local_funcid;
	data file  "func_effect.db"  contains func_effect;
	key  file  "func_effect.ix1" contains func_effect.id;
	key  file  "func_effect.ix2" contains func_effect.func_effect_funcid;
	data file  "f_inline.db"    contains f_inline;
	key  file  "f_inline.ix1"   contains f_inline.id;
	key  file  "f_inline.ix2"   contains f_inline.rva;
	data file  "structure.db"   contains structure;
	key  file  "structure.ix1"  contains structure.id;
	data file  "struct_member.db"   contains struct_member;
	key  file  "struct_member.ix1"  contains struct_member.id;
	key  file  "struct_member.ix2" contains struct_member.struct_member_struct;
	data file  "data_type.db"    contains data_type;
	key  file  "data_type.ix1"   contains data_type.id;
	key  file  "data_type.ix2"   contains data_type.name;
	data file  "constant.db"    contains constant;
	key  file  "constant.ix1"   contains constant.id;
	key  file  "constant.ix2"   contains constant.name;
	key  file  "constant.ix3"   contains constant.value;
	key  file  "constant.ix4"   contains constant.constant_group_value;
	data file  "const_group.db"    contains constant_group;
	key  file  "const_group.ix1"   contains constant_group.id;
	key  file  "const_group.ix2"   contains constant_group.name;
	data file  "bc_macro.db"       contains bc_macro;
	key  file  "bc_macro.ix1"      contains bc_macro.id;
	key  file  "bc_macro.ix2"      contains bc_macro.name;

	/* ADDRESS : A block of code or data in the file. */
	record address {
		ulong rva;			/* unique relative virtual address */
		ulong pa;			/* offset from file start [target.image] */
		ushort size;		/* number of bytes in this address object */
		ulong flags;			/* Flags such as code/data, etc */
		ulong dataType;   /* ID of data type for this addr [0=none] */
		int dataSize;     /* size of data [>1 == array of data_type] */
		ulong dataConst;	/* ID of CONSTANT rec for data addr [0=none] */
		ulong structure;	/* ID of STRUCTURE rec for this addr [0=none] */
		ulong comment;	/* ID of COMMENT rec for this addr [0=none] */
		ushort tag;		/* Application-specific data */

		primary key rva;
		alternate key pa;
	}

	/* SECTION : logical code, data, or header [info] sections of the file.
	             Each section begins with an ADDRESS record */
	record section {
		char name[32];		/* Name of Section */
		ulong rva;			/* Starting RVA */
		ulong size;			/* Size of Section */
		ulong flags;		/* Section Flags, e.g. rwx */
		ushort tag;			/* Application-specific data */

		primary key name;
		alternate key rva;
	}

	/* XREF : Cross references between ADDRESS records in the file. */
	record xref {
		ulong id;			/* Unique ID, from seq.xref */
		ulong from_rva;	/* "From" ADDRESS record */
		ulong to_rva;		/* "To" ADDRESS record */
		ulong type;			/* Type of reference, i.e. rwx */

		primary key id;
		alternate key from_rva; 
		alternate key to_rva; 
		alternate key xref_from_rva {from_rva, to_rva};
		alternate key xref_to_rva   {to_rva, from_rva};
	}

	/* CODE : A line of disassembled code tied to an ADDRESS record */
	record code {
		ulong rva;					/* RVA of ADDRESS record */
      	ulong func;          	/* ID of FUNCTION containing this CODE */
		char mnemonic[16];		/* Assembly language mnemonic */
		long dest;			   	/* Destination operand */
		long src;			   	/* Source operand */
		long aux;			   	/* Additional [auxiliary] operand */
		unsigned int mnemType;	/* Flags for mnemonic */
		unsigned int destType;	/* Flags for Destination */
		unsigned int srcType;	/* Flags for Source */
		unsigned int auxType;	/* Flags for Aux */
		ushort tag;				/* Application-specific data */

		primary key rva;
		alternate key code_function {func, rva}; 
		alternate key code_dest_op	    {dest, rva}; /* slow, useful */
		alternate key code_src_op      {src, rva};	/* ditto */	
		alternate key code_mnemonic {mnemonic, rva};
	}

	/* ADDR_EXP : address expression */
	record addr_exp {
      ulong id;
		int scale;
		int index;
		int base;
		long disp;
      ulong  flags;  /* 4  bytes: 1 for each of S,I,B, and disp */
		ulong name;	/* id of name for this expr, if applicable */
      
		primary key id;
	}

	/* CODE_EFFECT : Effects of a line of code on registers, etc */
	record code_effect {
		ulong id;            /* Unique ID from seq.code_effect */
		ulong rva;           /* rva of CODE record causing effects */
		int loc;             /* id of register being changed */
		ulong type;		   /* type of location (reg, memory...) */
		ulong change;          /* type of change */
		int amount;		   /* amount of change */

		primary key id;
		alternate key code_effect_rvaid {rva, id};	
	}

	/* REG_STATE : State of a single register, managed by vm.h */
	/*             These are saved by the VM */
	record reg_state {
		ulong id;			/* Unique ID from seq.reg_state */
		int reg;			/* index of register in VM regtable */
		ulong val;			/* data in register */
		int reftype;		/* sysref or xref */
		ulong ref;			/* id of XREF or SYSREF */

		primary key id;
		alternate key reg_state_by_ref {ref, reftype, reg};
	}

	/* SYSREF : Reference to OS construct, e.g. system call, trap, device */
	record sysref {
		ulong id;			/* Unique ID from seq.sysref */
		int type;			/* type of sysref, from os.h */
		int subtype;		/* internal type, e.g. syscall type */
		int ref;			/* number/internal ID of sysref */
		ulong rva;			/* CODE inst making reference */

		primary key id;
		alternate key sysref_type_id { type, ref, id };
		alternate key sysref_from { rva, type };
	}


	/* INT_CODE : Intermediate representation of a CODE record -- generated
	              as part of a decompilation pass */
	record int_code {
		ulong id;				/* unique ID from seq.int_code */
		ulong opcode;			/* intermediate instruction */
		ulong src;				/* 'src' [first] operand */
		ulong sType;
		ulong dest;				/* 'dest' [last] operand */
		ulong dType;
		ulong arg;				/* 'arg' [middle] operand */
		ulong aType;
		ulong func;				/* FUNCTION owning this INT_CODE */
		ulong order;			/* order in function */
		ulong rva;				/* ADDRESS associated with this */
		ulong comment;			/* COMMENT for this INT_CODE */
		ushort tag;				/* Application-specific data */

		primary key id;
		alternate key rva;
		alternate key fn_order { func, order };
	}

	/* ---------------------------------  decomp stuff from  int_code analysis

	EXP_ROOT : expression 'header'
	how to know where to print these in code?

	record exp_root {
		ulong id;
		ulong func;			// function owning this
		ulong order;		// order this occurs in func
		ulong int_code;		// int_code we are tied to
		ulong comment;
	}


	EXP_TREE_NODE
	EXP_TREE_ARG
	Operators:
	*,+,-,/,%,^,|,&,!,==,||,&&,.,++,--,**,//,%%,=

	EXP_TREE : node in an expression tree

	record exp_tree {
		ulong id;
		ulong root;			// exp_root owning this
		int op;
		int l_type; 		// type of 'left'
		int r_type;			// type of 'right'
		ulong left;
		ulong right;		// exp_tree || arg_list
	}

	ARG_REG (is reg id)
	ARG_IMM (is immediate value)
	ARG_NAME (is name ID)
	ARG_ADDR (is addr)
	ARG_ADDREX (is addr expr)

	ARG_LIST : arguments to expression/function
	record arg_list {
		ulong id;
		ulong root;			// root of arg_list
		ulong exp_tree;		// exp_tree owning this
		ulong d_type;		// data type
		ulong arg_type;		// is arg reg, addr, imm, etc
		ulong arg;			// value of arg (prob: float, negative num)
		ulong next;			// id of next arg_list node
	}

	CODE_BLOCK : block of expressions
	how to know where to print these in code?

	CODE_BLOCK_IF
	CODE_BLOCK_ELSEIF
	CODE_BLOCK_ELSE
	CODE_BLOCK_DO
	CODE_BLOCK_WHILE
	CODE_BLOCK_FOR
	CODE_BLOCK_SWITCH
	CODE_BLOCK_CASE
	CODE_BLOCK_END

	record code_block {
		ulong id;
		ulong root;			// parent code_block
		ulong type;			// above types
		ulong pre;			// exp_root s
		ulong post;
		ulong cond;
	}

	*/

	/* FIN_CODE : Final representation of a decompiled INT_CODE record */
	record fin_code {
		ulong id;				/* unique ID from seq.fin_code */
		ulong rva;				/* RVA where code starts */
		int size;				/* Size of code replaced by record */
		char line[256];		/* line of source code */
		ushort tag;			/* Application-specific data */

		primary key id;
		alternate key rva;
	}

	/* EXPORT_ADDR : Exported or 'public' symbols in the program */
	record export_addr {
		ulong rva;				/* RVA of exported code/data */

		primary key rva;
	}

	/* IMPORT_ADDR : Imported or 'library' functions in the program */
	record import_addr {
		ulong library;			/* ID of library [if known] */
		ulong type;				/* Import type [func, data] */
		ulong rva;				/* Address of relocation */

		primary key rva;
		alternate key import_library {library, rva};	
	}

	/* LIBRARY : Shared libraries required by the program */
	record lib {
		ulong id;					/* unique ID for each library */
		char name[64];			/* name of library file [soname] */
		int ver_hi;				/* version [major] */
		int ver_lo;				/* version [minor] */

		primary key id;
		alternate key name;
	}
		

	/* STRING : Arrays of characters [data ADDRESS records] */
	record string {
		ulong rva;				/* RVA of the ADDRESS record */
		int length;				/* length of string */
		char text[256];		/* ASCII representation of string */

		primary key rva;
	}

	/* NAME : Global names for ADDRESS records */
	/*        NOTE: a name is for anything with an rva -- not libs, etc */
	/*              or for 'anonmous' names [e.g. addr exprs, locals */
	record name {
		ulong id;				/* Unique ID from seq.name */
		ulong rva;				/* RVA of ADDRESS record */
		ulong type;				/* Type of name [auto, user, etc] */
		char text[64];		

		primary key id;
		alternate key rva;
		alternate key text;
	}

	/* COMMENT : Comment strings linked to ADDRESS or FUNCTION records */
	record comment {
		ulong id;			/* Unique ID from seq.comment */
		ulong type;			/* Type of comment[repeatable,auto,user] */
		char text[256];
		ushort tag;			/* Application-specific data */

		primary key id;
	}

	/* FUNCTION : A complete subroutine within the program */
	record function {
		ulong id;			/* Unique ID from seq.func */
		ulong rva;			/* RVA of ADDRESS object */
		ushort size;		/* Size of function */
		ulong ret_type;      	/* Return Type of function */
		ulong comment;		/* ID of COMMENT record for this function */
		ushort tag;			/* Application-specific data */

		primary key id;
		alternate key rva;
	}

	/* FUNC_PARAM : Represents parameters associated with a FUNCTION record */
	record func_param {
		ulong id;			/* Unique ID from seq.func_param */
		ulong func;			/* FUNCTION record that 'owns' the param */
		ulong type;			/* DATA_TYPE record [size, type of param] */
		int size;            /* 1 = normal, 2+ == array of data type */
		ulong addr_exp;		/* id of FP-based ADDR_EXP || reg code*/
		ulong flags;           /* flags : ADDR_EXP or REG ? */
		char name[32];			
		ulong const_group;	/* [opt] : grouping of values for param */

		primary key id;
		alternate key func_param_funcid {func, id};	
	}

	/* FUNC_LOCAL : Local variables associated with a FUNCTION record */
	record func_local {
		ulong id;			/* Unique ID from seq.func_local */
		ulong func;			/* FUNCTION record that 'owns' the var */
		ulong type;			/* DATA_TYPE record [size, type of var] */
		int size;            /* 1 = normal, 2+ == array of data type */
		ulong addr_exp;		/* id of FP-based ADDR_EXP || reg code*/
		ulong flags;           /* flags : ADDR_EXP or REG ? */
		char name[32];			

		primary key id;
		alternate key func_local_funcid {func, id};	
	}

	/* FUNC_EFFECT : Effects of the function registers, etc */
	record func_effect {
		ulong id;            /* Unique ID from seq.func_effect */
		ulong func;          /* FUNCTION record that 'owns' the effect */
		int loc;             /* id of register being changed */
		ulong type;		   /* type of location (reg, memory...) */
		ulong change;          /* type of change */
		int amount;		   /* amount of change */

		primary key id;
		alternate key func_effect_funcid {func, id};	
	}

	/* F_INLINE : A block of library code linked to an ADDRESS record; used to
	              mark blocks of repeated inline library code [e.g. memcpy]
			  so they can be hidden/folded like regular functions. An
			  ADDRESS record is associated with an F_INLINE, which 
			  contains details regarding size and a code template */
	record f_inline {
		ulong id;				/* Unique ID from seq.f_inline */
		char name[64];
		ulong rva;				/* Rva of 'template' code */
		ushort size;			/* Size of inline function */
		ushort tag;				/* Application-specific data */

		primary key id;
		alternate key rva;
	}

	/* STRUCTURE : A user-defined data structure which can be applied to an
	               ADDRESS record */
	record structure {
		ulong id;				/* Unique ID from seq.structure */
		char name[32];
		ushort size;			/* Total size of structure */

		primary key id;
	}

	/* STRUCT_MEMBER : Member data types for a given STRUCTURE record */
	record struct_member {
		ulong id;			/* Unique ID from seq.struct_member */
		ulong type;			/* ID of DATA_TYPE for this member */
		int size;			/* 1 = normal, 2+ == array of data type */
		ulong structure;		/* ID of STRUCTURE owning this member */
		int order;			/* Order of member in the struct [1,2,3] */
		char name[32];
		ulong const_group;	/* [opt] : grouping of values for field */

		primary key id;
		alternate key struct_member_struct {structure, order};	
	}
		
	/* DATA_TYPE : Name and size for predefined data or variable types --
	               like typedef */
	record data_type {
		ulong id;			/* Unique ID from seq.data_type */
		int size;			/* Size of data type */
		ulong flags;           /* signed/unsigned, pointer, etc */
		char name[32];		/* High-level name of data type 'long' */

		primary key id;
		alternate key name;
	}
		
	/* CONSTANT : Name and value for numeric constants, like #define */
	record constant {
		ulong id;				/* Unique ID from seq.constant */
		char name[32];			
		long value;				/* Value of constant */
		ulong group;			/* id of constant_group record */

		primary key id;
		alternate key name;
		alternate key value;
		alternate key constant_group_value { group, value };
	}
	
	/* CONSTANT_GROUP : Constant groupings, e.g. enumerations */
	record constant_group {
		ulong id;				/* Unique ID from seq.const_group */
		char name[32];

		primary key id;
		alternate key name;
	}

	/* BC_MACRO : A user-recorded or -scripted bc macro */
	record bc_macro {
		ulong id;		/* Unique ID from seq.bc_macro */
		ulong type;		/* 1: macro=script, 2: macro=name of .bc file */
		char name[32];
		char macro[2048];	/* bc Source Code */

		primary key id;
		alternate key name;
	}
}

/* end-of-file */

