/* api_system.c : part of The Bastard disassembly environment 
 *
 */
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <api/api_main.h>
#include <api/api_error.h>
#include <extension.h>
#include <config.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;

/*    API section:
 *
 *    17. System Interaction   :: Various utility functions
 */

/* ------------------------------------------------------System Interaction */
int sys_init(char *base)
{
	if ( ! env_set_base( base ) ) {
		return( sys_set_lasterr(5210) );
	}
	env_defaults();
	memset((char *) &target, 0, sizeof (struct DISASM_TGT));
	env_target_defaults();
	env_pref_defaults();
	DEBUG_Print("sys_initializing Interpreter\n");
	sys_init_interpreter();
	DEBUG_Print("sys_initializing RDBMS\n");
	if (! db_init() ) {
		return(0);	/* error set by db_init() */
	}
	/* set up DB linked list */
	db_head = db_current = db_target;
	db_target->db_id = -1;
	/*DEBUG_Print("Loading Config DB\n");
	   db_head = db_current = db_target->prev = db_config;
	   db_config->next = db_target;
	   db_config->db_id = db_target->db_id = -1;
	   LoadConfigDB(NULL); */
	DEBUG_Print("sys_initializing Extensions\n");
	sys_init_extensions();
	return( sys_re_init(base) );
}

int sys_re_init(char *base)
{
	char name[PATH_MAX];


	if (! base || ! base[0] ) { 
		return( sys_set_lasterr(5210) );
	}
	DEBUG_Print("Resetting Environment\n");
	/* set all structs to 0 and set defaults */
	memset(&target, 0, sizeof (struct DISASM_TGT));
	strncpy(disasm_env.base, base, PATH_MAX);

	env_defaults();
	env_target_defaults();
	env_pref_defaults();

	/* set up pseudovirtualmachine */
	DEBUG_Print("sys_initializing VM\n");
	vm_init();

	/* load default extensions */
	/* NOTE: this should be changed to use the target_set_* routines */
	DEBUG_Print("Loading Default Extensions\n");
	ext_gen_filename( "arch", target.arch.name, name, PATH_MAX );
	DEBUG_PrintVar(name, "%s");
	DEBUG_PrintVar(disasm_env.base, "%s");
	DEBUG_PrintVar(target.arch.name, "%s");
	load_extension(EXT_ARCH, name, ext_arch);
	ext_gen_filename( "asm", target.assembler.name, name, PATH_MAX );
	DEBUG_PrintVar(name, "%s");
	DEBUG_PrintVar(target.assembler.name, "%s");
	load_extension(EXT_ASM, name, ext_asm);
	ext_gen_filename( "formats", target.format.name, name, PATH_MAX );
	DEBUG_PrintVar(name, "%s");
	DEBUG_PrintVar(target.format.name, "%s");
	load_extension(EXT_FORMAT, name, ext_format);
	ext_gen_filename( "lang", target.lang.name, name, PATH_MAX );
	DEBUG_PrintVar(name, "%s");
	DEBUG_PrintVar(target.lang.name, "%s");
	load_extension(EXT_LANG, name, ext_lang);
	ext_gen_filename( "comp", target.comp.name, name, PATH_MAX );
	DEBUG_PrintVar(name, "%s");
	DEBUG_PrintVar(target.comp.name, "%s");
	load_extension(EXT_COMP, name, ext_comp);
	ext_gen_filename( "os", target.os.name, name, PATH_MAX );
	DEBUG_PrintVar(name, "%s");
	DEBUG_PrintVar(target.os.name, "%s");
	load_extension(EXT_OS, name, ext_os);

	/* load history for previous user */
	//DEBUG_Print("Loading User Profile\n");
	//user = getenv("USER");
	//LoadProfile(PROFILE_ENV, user);
	//LoadProfile(PROFILE_PREF, user);

	/* override everything with command-line options etc */
	disasm_prefs.options |= disasm_prefs.override;
	if (disasm_prefs.options & COLOR_TTY)
		disasm_env.output = ASM_OUTPUT_TTY_COLOR;

	return(1);
}


int sys_find_base(char *buf, int len, char **argv)
{
	int x;
	FILE *fTemp;
	char tmpBuf[128], *tmp, *ptr;
	struct stat tmpStat;

	/* + Get the location of bastard [for .bdb, libs, etc] */
	/* the key to our patented 'Run Anywhere' technology ;) */
	//tmp = getenv("BASTARD_HOME");
	tmp = getenv("BASTARD_BASE");
	if ( tmp ) {
		DEBUG_Print("Got BASTARD_BASE from env\n");
		strncpy(buf, tmp, len);
		return(1);
	} 

	/* okie-dokie, time to guess based on argv... */
	tmp = tmpBuf;
	if (strchr(argv[0], '/') != NULL) {
		DEBUG_Print("Absolute path to bastard\n");
		strncpy(tmp, argv[0], sizeof (tmpBuf) - 1);
	} else {
		DEBUG_Print("Using which to guess base dir\n");
		fTemp = popen("which bastard", "r");
		if (!fTemp) {
			fprintf( stderr,
				"Which is broken! Use Explicit Path Names!\n");
			return (0);
		} else {
			fgets(tmp, sizeof (tmpBuf), fTemp);
			pclose(fTemp);
			x = strlen(tmp) - 1;
			while (isspace(tmp[x])) {
				/* strip carriage return and other nonsense */
				tmp[x] = 0;
				x--;
			}
		}
	}
	/* handle symbolic links to the bastard */
	if (lstat(tmp, &tmpStat) == -1) {
		fprintf(stderr, "Fatal error %s: lstat %s\n", strerror(errno),
			       tmp);
		return (0);
	}
	if (S_ISLNK(tmpStat.st_mode)) {
		DEBUG_Print("Symbolic link to the bastard\n");
		if (readlink(tmp, tmp, 128) == -1) {
			fprintf(stderr, "Fatal Error %X: readlink %s\n", errno, tmp);
			return (0);
		}
	}

	if (len < strlen(tmp)) {
		fprintf( stderr, 
		    "Fatal Error : %d byte buffer is too small for path '%s'\n",
		     len, tmp);
		return (0);
	}

	/* now, we have the final location for the executable */
	buf[0] = 0;
	/* strip program name from path */
	ptr = strrchr(tmp, '/');
	ptr[0] = 0;
	ptr = strrchr(tmp, '/');
	if ( ! ptr )  	ptr = tmp;
	else 			ptr++;

	/* NOTE: all binaries must be in $BASTARD_BASE/bin for this to work */
	/*       We assume we are in $BASTARD_BASE/bin and set $BASTARD_SHARE
	 *       to $BASTARD_BASE/share/bastard $BASTARD_INCLUDE to 
	 *       $BASTARD_BASE/include and $BASTARD_MAN to $BASTARD_BASE/man
	 */
	if ( ! strcmp( ptr, "bin") ) {
		if ( ptr == tmp ) {
			/* tmp is just "bin" */
			ptr[0] = 0;
		} else {
			*(ptr - 1) = 0;
		}
	}
	/* is this an absolute path? */
	if ( tmp[0] != '/' )  {
		/* no! make it so, then ! */
		getcwd( buf, len );
		if ( tmp[0] && (tmp[0] != '.' || tmp[1] == '.') ){
			strncat( buf, "/", len - strlen(buf) - 1);
			strncat( buf, tmp,  len - strlen(buf));
		}
	} else {
		strncat( buf, tmp,  len - strlen(buf));
	}
	return(1);
}
void sys_msg(const char *str)
{
/* Print a message unless QUIET is flag is set 
 */
	if (!(disasm_prefs.options & QUIET))
		printf("%s", str);
	return;
}

void sys_errmsg(int num, const char *str)
{
/* Print a message and error code, regardless of QUIET */
	fprintf(stderr, "Error %d: %s\n", num, str);
	return;
}

void sys_quit(void)
{
	char *user = getenv("USER");

	DEBUG_Print("sys_quitting...\n");
	/* disabled until flags are working everywhere 
	   if ( disasm_env.flags )  */
	if (target.image) {
		target_close_db();
	}
	/* DEBUG_Print("Updating user profile\n");
	   //UpdateProfile(PROFILE_ENV, user);
	   //UpdateProfile(PROFILE_PREF, user);
	   UnloadConfigDB(); */
	vm_cleanup();
	DEBUG_Print("Cleaning Up Extensions\n");
	cleanup_extensions();
	env_set_flag(QUIT_NOW);
}

int sys_exec(char *args)
{
	char buf[256];
	sprintf(buf, ".bdb/%s %s", target.info.name, args);
	return (system(buf));
}

int sys_debug(void)
{
	char buf[256];
	sprintf(buf, "%s .bdb/%s", disasm_prefs.debugger, target.info.name);
	return (system(buf));
}

int sys_core(void)
{
/* start target in new process w/ core dumps enabled */
/* wait for user to hit key */
/* send a sig_core to target PID */
	sys_msg("sys_core not yet supported!\n");
	return (-1);
}

int sys_panic(void)
{
/* Similar to 'core' */
/* Get a kernel dump w/o killing the kernel, if possible */
/* ... the idea is to get a snapshot of the kernel as a target is
 * running */
	sys_msg("sys_panic not yet supported!\n");
	return (-1);
}

/* ----------------------------------------------- Error Handling */
int _last_bastard_error;

int sys_set_lasterr(int errnum)
{
/* This returns 0 so that you can do a `return(sys_set_lasterr(#));` */
	_last_bastard_error = errnum;
	return (0);
}

char *sys_get_errstr(int errnum)
{
	int x;

	for (x = 0; _bastard_error_table[x].errnum != 0; x++)
		if (_bastard_error_table[x].errnum == errnum)
			return (_bastard_error_table[x].errstr);
	return (NULL);
}

int sys_get_lasterr()
{
	return (_last_bastard_error);
}

char *sys_lasterr_str()
{
	return (sys_get_errstr(_last_bastard_error));
}

void sys_print_errmsg(int errnum)
{
	char *msg;

	sys_errmsg(errnum, sys_get_errstr(errnum));
	return;
}
void sys_sprint_errmsg(char *buf, int errnum)
{
	if (!buf)
		return;
	strcpy(buf, sys_get_errstr(errnum));
	return;
}
