/* api.c : part of The Bastard disassembly environment ========================G
 *
 * Contains  the bulk of the internal code -- specifically, the API routines
 * provided for internal and scripting use. Functions should be in the same
 * order as in api.h
 */
#include <api/api_main.h>
#include <extension.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;

/* API Sections:
 *
 *     7. Strings               :: Creating/manipulating NAME objects
 */


/* ---------------------------------------------------------------Strings */
int str_make_printable(char *name, int len)
{
	/* quick function to get rid of spaces, etc in names */
	int x;
	for (x = 0; x < len; x++) {
		if ((name[x] < '0' || name[x] > '9') &&
		    (name[x] < 'A' || name[x] > 'Z') &&
		    (name[x] < 'a' || name[x] > 'z') &&
		    (name[x] != '_' && name[x] != '-')) {	/* add user ctrl later */
			/* replace all unapproved chars with '_' */
			name[x] = '_';
		}
	}
	return (1);
}
int str_get(long rva, char *buf, int len)
{
	/* returns length of string, *not* bytes written */
	struct string s;

	if (d_keyfind(STRING_RVA, &rva) != S_OKAY)
		return (sys_set_lasterr(5000));
	d_recread(&s);
	strncpy(buf, s.text, len);
	return (s.length);
}

int str_get_raw(long rva, char **buf)
{
	struct string s;
	struct address a;
	char *ptr, *dest;
	int x = 0, y = 0;

	if ( bdb_index_find(STRING_RVA, &rva, &s) &&
		bdb_index_find(ADDRESS_RVA, &rva, &a) ) {
		if (a.pa > target.info.size)
			return (0);
		ptr = target.image + a.pa;
		*buf = calloc(s.length, 2);	/* need room for excape chars */
		if (!*buf)
			return (0);
		for (x = 0; x < s.length; x++) {
			dest = *buf + y;
			if (ptr[x] == '\n') {
				dest[0] = '\\';
				dest[1] = 'n';
				y++;
			} else if (ptr[x] == '\r') {
				dest[0] = '\\';
				dest[1] = 'r';
				y++;
			} else if (ptr[x] == '\t') {
				dest[0] = '\\';
				dest[1] = 't';
				y++;
			} else
				*dest = ptr[x];
			y++;
		}
	}
	return (x);
}

/* TODO: Have this resize addr so it matches string size */
int str_new(long rva)
{
	struct string s;
	struct address a;

	if (! bdb_index_find(ADDRESS_RVA, &rva, &a) )
		return (sys_set_lasterr(4020));
	if (a.pa < target.info.size) {
		s.length =
		    ext_makestr(target.image + a.pa, s.text, 256, STR_STD);
	} else
		return (sys_set_lasterr(4040));
	s.rva = rva;
	//s.length = strlen(s.text);
	a.size = s.length;	/* this is not wholly accurate: no term char :) */
	bdb_record_update(ADDRESS_RVA, &rva, &a);
	if (d_keyfind(STRING_RVA, &rva) == S_OKAY)
		d_recwrite(&s);
	else if (d_fillnew(STRING, &s) != S_OKAY)
		return (sys_set_lasterr(db_error()));
	return (s.length);
}

int str_print()
{
	struct string s;

	d_keyfrst(STRING_RVA);
	while (db_status == S_OKAY) {
		d_recread(&s);
		printf("%X\t%s\n", s.rva, s.text);
		d_keynext(STRING_RVA);
	}
	return (1);
}

int str_find_in_range(long rva, int size, int type)
{
	/* rva : start of range   size: extent of range  type: string type */
	struct address a = {0};
	struct string str = {0};
	char *ptr, name[64];
	int pos = 0, len, *state;
	long pa, str_pa;

	state = db_save_state();
	/* 1. Get offset into file image to search for strings */
	if ( rva && bdb_index_find(ADDRESS_RVA, &rva, &a) ) {
		pa = a.pa;
	} else if (rva < target.info.size) {
		pa = rva;
	} else {
		pa = 0;
	}

	if (pa + size >= target.info.size){
		db_restore_state(state);
		return (sys_set_lasterr(4040));
	}

	/* 2. Call FindString in language module until 0 is returned */
	len = size;
	ptr = &target.image[pa];
	while (len > 0 && (str.length =
			   ext_findstr(ptr, len, &pos, str.text, 256, type))) {
		str.rva = rva + (((int)ptr) - (int)(&target.image[pa])) + pos;
		str_pa = pa + (((int)ptr) - (int)(&target.image[pa])) + pos;

		/* ignore string if one already exists */
		if (d_keyfind(STRING_RVA, &str.rva) != S_OKAY) {
			d_fillnew(STRING, &str);
 
			/* resize addr */
			addr_new(str.rva, str.length, str_pa, 0);
			/* create name if none exists */
			if ( d_keyfind(NAME_RVA, &str.rva) != S_OKAY ) {
				snprintf(name, 63, "str_%s", str.text);
				str_make_printable(name, strlen(name));
				name_new(str.rva, name, NAME_AUTO);
			}
		}
		/* adjust size of buffer */
		len -= (pos + str.length);
		ptr += (pos + str.length);
		ptr = &target.image[str_pa + str.length];
	}

	db_restore_state(state);
	return (1);
}
