#include <api/api_main.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;

/* API sections:
 *    14. Names   g            :: User-defined address names
 *    15. Comments   g         :: User-defined address comments
 */



/* ------------------------------------------------------ utility routines */
void name_validate(char *name)
{
	/* remove all weird characters from name */
	int x;

	for (x = 0; x < strlen(name); x++) {
		if (name[x] && (name[x] < 0x20 || name[x] > 0x7E))
			name[x] = '_';	/* mangling char ;) */
	}
	return;
}
/* ---------------------------------------------------------------Names */
int name_new(long rva, char *name, int type)
{
	struct name n = {0}, n_old;

	if (! name) return(0);
	n.rva = rva;
	n.type = type;
	strncpy(n.text, name, 63);
	n.text[63] = 0;

	name_validate(n.text);

	if ( d_keyfind(NAME_RVA, &rva) == S_OKAY) {
		d_recread( &n_old );
		n.id = n_old.id;
		d_recwrite(&n);
	} else {
		n.id = ++seq.name;
		d_fillnew(NAME, &n);
	}

	if (db_status != S_OKAY) {
		return( sys_set_lasterr(db_error()) );
	}

	disasm_env.flags |= DB_MOD;
	return (n.id);
}

int name_new_default( unsigned long rva, int type ) 
{
	char buf[64], *prefix;

	switch (type) {
		case NAME_NEWSUB:
			prefix = "sub"; 	break;
		case NAME_NEWNEAR:
			prefix = "@@"; 	break;
		case NAME_NEWPTR:
			prefix = "ptr"; 	break;
		case NAME_NEWARY:
			prefix = "ary";	break;
		case NAME_NEWSTR:
			prefix = "str";	break;
		case NAME_NEWSEC:
			prefix = "sec";	break;
		case NAME_NEWLOC:
		default:
			prefix = "loc";
	}
	snprintf(buf, 64, "%s_%08X", prefix, rva);
	return( name_new(rva, buf, NAME_AUTO) );
}

int name_get(long addr, char *buffer)
{
	struct name n;

	if (d_keyfind(NAME_RVA, &addr) == S_OKAY) {
		d_recread(&n);
		strcpy(buffer, n.text);
		return (strlen(n.text));
	}
	return (sys_set_lasterr(4710));
}

int name_get_type(long addr)
{
	struct name n;

	if (d_keyfind(NAME_RVA, &addr) == S_OKAY) {
		d_recread(&n);
		return (n.type);
	}
	return (sys_set_lasterr(4710));
}


/* delete name for 'addr' */
int name_del(long addr)
{
	if (d_keyfind(NAME_RVA, &addr) == S_OKAY) {
		d_delete();
		return (1);
	}
	return (sys_set_lasterr(4710));
}

/* ---------------------------------------------------------------Comments */
int comment_get(int id, char *buf)
{
	/* comments are handled by ID, not by their text contents */
	struct comment c;

	if (!buf)
		return (sys_set_lasterr(9010));
	if (d_keyfind(COMMENT_ID, &id) == S_OKAY) {
		d_recread(&c);
		strcpy(buf, c.text);
		return (strlen(buf));
	}
	return (sys_set_lasterr(4720));
}

int comment_new(char *buf, int type)
{
	struct comment c;

	if (!buf)
		return (sys_set_lasterr(9010));
	c.id = ++seq.comment;
	c.type = type;
	strncpy(c.text, buf, 255);
	c.text[255] = '\0';
	name_validate(c.text);
	d_fillnew(COMMENT, &c);
	if (db_status == S_OKAY)
		return (c.id);
	db_error();
	return (0);
}

int comment_change(int id, char *buf)
{
	struct comment c;

	if (!buf)
		return (sys_set_lasterr(9010));
	if (id && d_keyfind(COMMENT_ID, &id) == S_OKAY) {
		d_recread(&c);
		strncpy(c.text, buf, 255);
		c.text[255] = '\0';
		name_validate(c.text);
		d_recwrite(&c);
	} else {
		c.id = ++seq.comment;
		c.type = CMT_USER;
		strncpy(c.text, buf, 255);
		c.text[255] = '\0';
		name_validate(c.text);
		d_fillnew(COMMENT, &c);
	}
	if (db_status == S_OKAY)
		return (1);
	db_error();
	return (0);
}

int comment_del(int id)
{
	if (!bdb_record_delete(COMMENT_ID, &id))
		return (sys_set_lasterr(4730));
	disasm_env.flags |= DB_MOD;
	return (1);
}
