#include <api/api_main.h>
#include <extension.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;


/* API sections:
 *     1. Loading             :: Loading files, DBs, and so on
 *     2. Saving               :: Saving DBs, asm listings, patched files
 */


int target_fix_path( char *path, char *buf, int len ) {
	// fix "~" if user specified it
	if (path[0] == '~') {
		if ( path[1] != '/' ) {
			snprintf( buf, len, "/home/%s", &path[1] );
		} else {
			snprintf( buf, len, "%s/%s", getenv("HOME"), &path[2] );
		}
	} else if (path[0] == '.' && path[1] != '.') {
		path++;

		getcwd(buf, len);
		strncat(buf, path, len - strlen(buf) - 1);
	} else {
		strncpy( buf, path, len );
	}
	return(1);
}

int target_path_info(char *path)
{
	char *c;

	/*    make target.info.path and target.info.name */
	c = strrchr(path, '/');
	if (c) {
		strncpy(target.info.path, path, c - path);
		strncpy(target.info.name, (c + 1), sizeof (target.info.name));
		target.info.path[c - path + 1] = 0;	/* terminate string */
	} else {
		getcwd(target.info.path, PATH_MAX);
		strncpy(target.info.name, path, sizeof (target.info.name));
	}
	return (1);
}

int target_new()
{
	char buf[PATH_MAX];
	FILE *tmp;
	struct stat tmpStat;

	/*   Copy target binary to .bdb */
	sprintf(buf, "cp -f %s/%s %s/.%s.bdb/%s", target.info.path,
		target.info.name, disasm_env.dbpath, target.info.name,
		target.info.name);
	if (!((tmp = popen(buf, "r")) && (pclose(tmp) == 0))) {
		return (sys_set_lasterr(4540));
	}

	/*   memory map target image */
	sprintf(buf, "%s/.%s.bdb/%s", disasm_env.dbpath, target.info.name,
		target.info.name);
	target.fd = open(buf, O_RDWR);
	if (target.fd == -1) {
		sprintf(buf, "Cannot open file %s", strerror(errno));
		sys_errmsg(9030, buf);
		return (sys_set_lasterr(9030));
	}
	disasm_env.flags |= TGT_MAPPED;
	fstat(target.fd, &tmpStat);
	target.info.size = tmpStat.st_size;
	target.image = mmap(0, target.info.size, PROT_READ | PROT_WRITE,
			    MAP_SHARED, target.fd, 0);

	if (target.image == (void *) -1) {
		target.image = 0;
		close(target.fd);
		target.fd = 0;
		return (sys_set_lasterr(9030));
	}
	/* I think we do not need this, if sections are created */
	/* should we leave it in for "stability"'s sake? */
	/*    Create a dummy address in DB to represent entire file */
	//addr_new(0, target.info.size, 0, ADDR_DATA);

	disasm_env.flags |= DB_LOADED || DB_MOD;
	/* clear all flags but extensions [user may have hand-loaded ext's] */
	target.status &= 0x0FF0;
	target.status |= DISASM_TGT_LOADED;
	return (1);
}

int target_make_bdb()
{
	struct stat tmpStat;
	char buf[PATH_MAX];

	sprintf(target.info.dbname, "%s.bdb", target.info.name);
	/* If target.bdb or .target.bdb exists, error out -- user must load these manually */

	sprintf(buf, "%s/.%s", disasm_env.dbpath, target.info.dbname);
	if (!stat(buf, &tmpStat)) {
		/* note: this can be recovered with target_load_bdb */
		return( sys_set_lasterr(4502) );
	}
	/* If $DB_PATH/target.bdb exists, use that instead */
	sprintf(buf, "%s/%s", disasm_env.dbpath, target.info.dbname);
	if (!stat(buf, &tmpStat)) {
		return( sys_set_lasterr(4501) );
	}

	/* from here on we are making a new DB */
	if (!db_load("bdb", target.info.dbname, db_target))
		return (0);	/* retain Last Error set by db_load */
	db_switch(db_target->db_id);

	/* do all the new target stuff */
	return (target_new());
}

/* ---------------------------------------------------------------Loading */
/* MMapping might cause problems with larger files; however it 
 * should have smaller storage requirements than actually loading
 * the bytes into the database itself. It might be better to mmap each
 * section, thus junk like .bss can be summarized instead of stored.
 */
int target_load(char *path) {
	struct stat tmpStat;
	char tgt_path[PATH_MAX];

	DEBUG_PrintMsg("Loading TARGET filename: %s\n", path);

	/* Make sure no DB is currently loaded */
	if (disasm_env.flags)
		target_close_db();

	/* fill target.info.name and target.info.path */
	if (stat(path, &tmpStat)) {
		sys_print_errmsg(4500);
		return (sys_set_lasterr(4500));
	}

	target_fix_path( path, tgt_path, PATH_MAX );
	target_path_info(tgt_path);

	/* load existing BDB or create new one for new target */
	return (target_make_bdb());
}

int target_load_bdb(char *dbname)
{
/* Load a previously-saved .bdb */
	struct stat s;
	FILE *f;
	FILE *tmp;
	int idx;
	char *pos, dbname_buf[PATH_MAX], buf[256];

	// Make sure no DB is currently loaded 
	if (disasm_env.flags & DB_LOADED)
		target_close_db();

	target_fix_path( dbname, dbname_buf, PATH_MAX );

	//  Load file header
	if (!target.info.dbname[0]) {
		pos = strrchr(dbname, '/');
		idx = (pos == 0) ? 0 : pos - dbname + 1;
		strncpy(target.info.dbname, &dbname[idx], PATH_MAX);
	}

	if (! db_load("bdb", dbname_buf, db_target))	return(0);
	db_switch(db_target->db_id);

	sprintf(buf, "%s/.%s/header.txt", disasm_env.dbpath,
		target.info.dbname);
	if (!stat(buf, &s)) {
		f = fopen(buf, "r");
		if (f) {
			target.info.header = calloc(s.st_size, 1);
			fread(target.info.header, s.st_size, 1, f);
			fclose(f);
		}
	}
	// Fill Target Struct
	sprintf(buf, "%s/.%s/.info", disasm_env.dbpath, target.info.dbname);
	f = fopen(buf, "r");

	if (f) {
		fscanf(f, "%s\n%s\n%x\n%lx\n%x\n%x\n%s\n%s\n%f\n%s",
			target.info.name, target.info.path, &target.info.size,
			&target.info.entry, &target.info.endian, &target.info.type,
			target.info.vendor, target.info.model, &target.info.version,
			target.info.comment);
		fscanf(f, "%s\n%s\n%s\n%s\n%s\n%s\n%x\n",
			target.arch.name, target.assembler.name, target.comp.name,
			target.format.name, target.lang.name, target.os.name,
			&target.status);
		fclose(f);
	} else
		return (sys_set_lasterr(4561));


	sprintf(buf, "%s/.%s/%s", disasm_env.dbpath, target.info.dbname,
		target.info.name);
	if (stat(buf, &s) && target.info.size != s.st_size) {
		sys_errmsg(9030, "Bad .bdb file! \n");
		target_close_db();
		return (sys_set_lasterr(9030));
	}


	target.fd = open(buf, O_RDWR);
	if (target.fd == -1) {
		sprintf(buf, "Cannot open file %s", errno);
		sys_errmsg(9030, buf);
		return (sys_set_lasterr(9030));
	}

	target.image = mmap(0, target.info.size, PROT_READ | PROT_WRITE,
			    MAP_SHARED, target.fd, 0);
	if (target.image == (void *) -1) {
		target.image = 0;
		return (sys_set_lasterr(9030));
	}
	disasm_env.flags |= TGT_MAPPED;

	if (!target.info.header) {
		target.info.header = calloc(64, 1);
		sprintf(target.info.header, "Header not available!\n");
	}
	if (!target.info.dbname[0])
		strncpy(target.info.dbname, dbname, PATH_MAX);

	disasm_env.flags |= DB_LOADED;
	return (1);
}

/* ---------------------------------------------------------------Saving */
void target_save_db_as(char *name)
{
/* store database and image in .bdb file */
	char buf[256], pwd[PATH_MAX];
	FILE *f, *tmp;

	sprintf(buf, "Saving DB to %s...\n", name);
	sys_msg(buf);
/* create .info file */
	sprintf(buf, "%s/.%s/.info", disasm_env.dbpath, target.info.dbname);
	DEBUG_PrintVar(buf, "%s");
	f = fopen(buf, "w");
	if (f) {
		fprintf(f, "%s\n%s\n%x\n%lx\n%x\n%x\n%s\n%s\n%f\n%s",
			target.info.name, target.info.path, target.info.size,
			target.info.entry, target.info.endian, target.info.type,
			target.info.vendor, target.info.model, target.info.version,
			target.info.comment);
		fprintf(f, "%s\n%s\n%s\n%s\n%s\n%s\n%x\n",
			target.arch.name, target.assembler.name, target.comp.name,
			target.format.name, target.lang.name, target.os.name,
			target.status);
		fclose(f);
	} else {
		sys_errmsg(4561, "Unable to create .info file!");
	}
/* create header str file */
	sprintf(buf, "%s/.%s.bdb/header.txt", disasm_env.dbpath,
		target.info.name);
	DEBUG_PrintVar(buf, "%s");
	f = fopen(buf, "w");
	if (f) {
		fprintf(f, "%s", target.info.header);
		fclose(f);
	}
	msync(0, target.info.size, MS_SYNC);
	getcwd(pwd, PATH_MAX);
	chdir(disasm_env.dbpath);

	sprintf(buf, "tar --remove-files -zcf %s .%s.bdb -C%s", name,
		target.info.name, disasm_env.dbpath);
	DEBUG_PrintVar(buf, "%s");
	if (!((tmp = popen(buf, "r")) && (pclose(tmp) == 0))) {
		unlink(name);	/* does this remove all contents of subdir? */
		chdir(pwd);
		sys_errmsg(4565, "Unable to save target .bdb!");
		sys_set_lasterr(4565);
		return;
	} else
		sys_msg("DB saved\n");
	chdir(pwd);
}

void target_save_bak()
{
	char fname[256];
	sprintf(fname, "./%s.%d.bdb", target.info.name, time(NULL));
	target_save_db_as(fname);
	return;
}

void target_save_db()
{
	char *str, info_file[256];
	struct stat statbuf;
	FILE *f;
	int c;

	if (disasm_prefs.options & DONT_SAVE)	return;
	if (disasm_prefs.options & ANNOY_USER) {
		printf("Save %s database? <Y/n/a/b/?> ", target.info.name);
		c = getc(stdin);
		getc(stdin);	//strip CR

		switch (c) {
		case 'n':
		case 'N':
			return;
		case 'b':
		case 'B':
			target_save_bak();
			return;
		case 'a':
		case 'A':
			printf("Save as: ");
			fgets(str, 256, stdin);
			if (str[strlen(str)] == '\n')
				str[strlen(str)] = '\0';
			strncpy(target.info.dbname, str, 255);
			free(str);
			break;
		case 'y':
		case 'Y':
		case '\n':
			break;
		case '?':
		case 'h':
		default:
			printf
			    ("  Y='Yes' N='No' A='save As...' B='make Backup'\n");
			target_save_db();
			return;
		}
	}
	target_save_db_as(target.info.dbname);
	disasm_env.flags ^= DB_MOD;
	return;
}

void target_close_db()
{
	char buf[128];
	FILE *tmp;


	DEBUG_Print("target_close_db: Saving DB\n");
	/* the if is disabled until DB_MOD is used everywhere again :) _m */
	//if (disasm_env.flags & DB_MOD) {         
	target_save_db();
	//   disasm_env.flags ^= DB_MOD;
	//}
	DEBUG_Print("target_close_db: Unloading DB\n");
	if (disasm_env.flags & DB_LOADED) {
		db_unload(db_target);
		disasm_env.flags ^= DB_LOADED;
	}
	DEBUG_Print("target_close_db: UnMapping Target File\n");
	if (disasm_env.flags & TGT_MAPPED) {
		close(target.fd);	/* cleanup mmap'ed image */
		munmap(target.image, target.info.size);
		target.image = NULL;
		disasm_env.flags ^= TGT_MAPPED;
	}
	if (target.info.header)
		free(target.info.header);
//      DEBUG_Print("target_close_db: Switch DB\n");
//      db_switch(db_config->db_id);
	DEBUG_Print("target_close_db: rm -rf'ing target dir\n");
	sprintf(buf, "rm -rf %s/.%s.bdb", disasm_env.dbpath, target.info.name);
	DEBUG_PrintVar(buf, "%s");
	if (!((tmp = popen(buf, "r")) && (pclose(tmp) == 0))) {
		sys_errmsg(4511, "Error cleaning up .bdb");
	}
	/* refill target struct with defaults */
	memset((char *) &target, 0, sizeof (struct DISASM_TGT));
	env_target_defaults();

	if (!(disasm_env.flags & QUIT_NOW)) {
		DEBUG_Print("target_close_db:  sys_re_init\n");
		sys_re_init(disasm_env.base);
	}
	DEBUG_Print("target_close_db: done\n");
	return;
}

int target_save_asm(char *filename)
{
	return( target_output_as("asm", filename, 0) );
}

int target_save_hll(char *filename)
{
	return( target_output_as("hll", filename, 0) );
}

int target_save_lst(char *filename)
{
	return( target_output_as("lst", filename, 0) );
}

int target_save_hex(char *filename)
{
	return( target_output_as("hex", filename, 0) );
}

int target_save_diff(char *filename)
{
	return( target_output_as("diff", filename, 0) );
}
int target_save_binary(char *filename)
{
	return( target_output_as("binary", filename, 0) );
}

int target_set_ext(char *format, char *arch, char *comp, char *os,
		   char *lang, char *asmblr)
{
	int rv = 1;
	if (format[0])
		rv = target_set_format(format, 0);
	if (rv && arch[0])
		rv = target_set_arch(arch, 0);
	if (rv && comp[0])
		rv = target_set_comp(comp, 0);
	if (rv && os[0])
		rv = target_set_os(os, 0);
	if (rv && lang[0])
		rv = target_set_lang(lang, 0);
	if (rv && asmblr[0])
		rv = target_set_asm(asmblr, 0);
	return (rv);
}

int target_set_format(char *file_format, int options)
{
/* Invoke the parser in $BASTARD/formats
 * This parser should parse the target file and create sections for
 * all data and code sections/segments in the file. Additional info,
 * such as relocations/imports/exports, should be dealt with by the
 * parser as well.
 */
	char libname[PATH_MAX] = {0};
	struct stat statbuf;

	DEBUG_PrintMsg("Setting TARGET format:  %s\n", file_format);
	if ( ! ext_gen_filename( "formats", file_format, libname, PATH_MAX) )
		return(sys_set_lasterr(4594));

	if ( ext_format->ext.filename && 
	     ! strcmp(libname, ext_format->ext.filename)) 
		return(1);

	DEBUG_PrintVar(libname, "%s");
	ext_format->options = options;
	if (!load_extension(EXT_FORMAT, libname, ext_format))
		return (sys_set_lasterr(4594));

	strncpy(target.format.name, file_format, 32);
	target.status |= DISASM_TGT_FORMAT_SET;
	return (1);
}

int target_apply_format()
{
	if (! ext_format->ext.filename[0] ) 
		return( sys_set_lasterr(4594) );
	/* call the read_header routine */
	ext_read_header();
	disasm_env.flags |= DB_MOD;
	/* move this into a master PhasesOfDisassembly controller */
	target.status |= DISASM_TGT_PREDISASM;
	return (1);

}

int target_set_arch(char *file_arch, int options)
{
	char libname[PATH_MAX];
	struct stat tmpStat;

	/* default to the current disassembler */
	if (file_arch == NULL)
		strcpy(libname, ext_arch->ext.filename);
	else if ( ! ext_gen_filename( "arch", file_arch, libname, PATH_MAX) )
		return(sys_set_lasterr(4591));

	/* If the disassembler isn't loaded, load it */
	ext_arch->options = options;
	if (!load_extension(EXT_ARCH, libname, ext_arch))
		return (sys_set_lasterr(4291));

	strncpy(target.arch.name, file_arch, 32);
	target.info.endian = ext_arch->endian;
	target.status |= DISASM_TGT_ARCH_SET;
	return (1);
}
int target_set_asm(char *asm_output, int options)
{
	char libname[PATH_MAX];
	struct stat statbuf;

	DEBUG_PrintMsg("Setting TARGET assembler: %s\n", asm_output);
	if ( ! ext_gen_filename( "asm", asm_output, libname, PATH_MAX) )
		return(sys_set_lasterr(4592));

	if (ext_asm->ext.filename && ! strcmp(libname, ext_asm->ext.filename)) 
		return(1);
	ext_asm->options = options;
	if (!load_extension(EXT_ASM, libname, ext_asm))
			return (sys_set_lasterr(4592));

	/* this mechanism will be replaced by something better.... */
	if (env_get_opt_flag(COLOR_TTY)) {
		env_tty_asm(ext_asm->asm_ttyColor);
		env_tty_data(ext_asm->data_ttyColor);
	}
	strncpy(target.assembler.name, asm_output, 32);
	target.status |= DISASM_TGT_ASM_SET;
	return (1);
}

int target_set_comp(char *compiler, int options)
{
	char libname[PATH_MAX];
	struct stat statbuf;

	DEBUG_PrintMsg("Setting TARGET Comp: %s\n", compiler);
	if ( ! ext_gen_filename( "comp", compiler, libname, PATH_MAX) )
		return(sys_set_lasterr(4597));

	if (ext_comp->ext.filename && ! strcmp(libname, ext_comp->ext.filename))
		return (1);
	ext_comp->options = options;
	if (!load_extension(EXT_COMP, libname, ext_comp))
		return (sys_set_lasterr(4597));

	/* TODO: need some method of determining target bitness */
	strncpy(target.comp.name, compiler, 32);
	ext_add_data_types(4);
	target.status |= DISASM_TGT_COMP_SET;
	return (1);
}

int target_set_os(char *os, int options)
{
	char libname[PATH_MAX];
	struct stat statbuf;

	DEBUG_PrintMsg("Setting TARGET os: %s\n", os);
	if ( ! ext_gen_filename( "os", os, libname, PATH_MAX) )
		return(sys_set_lasterr(4598));

	if (ext_os->ext.filename && ! strcmp(libname, ext_os->ext.filename))
		return (1);
	ext_os->options = options;
	if (!load_extension(EXT_OS, libname, ext_os))
		return (sys_set_lasterr(4598));

	strncpy(target.os.name, os, 32);
	target.status |= DISASM_TGT_OS_SET;
	return (1);
}

int target_set_lang(char *language, int options)
{
	char libname[PATH_MAX];
	struct stat statbuf;

	DEBUG_PrintMsg("Setting TARGET Lang: %s\n", language);
	if ( ! ext_gen_filename( "lang", language, libname, PATH_MAX) )
		return(sys_set_lasterr(4595));

	if (ext_lang->ext.filename && ! strcmp(libname, ext_lang->ext.filename))
		return (1);

	ext_lang->options = options;
	if (!load_extension(EXT_LANG, libname, ext_lang))
		return (sys_set_lasterr(4595));

	strncpy(target.lang.name, language, 32);
	target.status |= DISASM_TGT_LANG_SET;
	return (1);
}

void * plugin_load(char *name, int options)
{
	char libname[PATH_MAX];
	struct stat statbuf;
	void * plugin;

	if (! name )	return(0);
	/* if .bc not exist, assume .so */
	if ( ! ext_gen_filename( "plugins", name, libname, PATH_MAX) )
		return((void *)sys_set_lasterr(4596));

	plugin = ext_plugin_load( libname, options );
	
	if (! plugin)
		return ( (void *) (sys_set_lasterr(4596)));

	return (plugin);
}

int plugin_exec_main(void *plugin, void *param)
{
	if (! plugin) 	return(0);
	return (ext_plugin_main(plugin, param));
}

int plugin_exec( void *plugin, char *name, void *param ) {
	if (! plugin || ! name) 	return(0);
	return (ext_plugin_exec( plugin, name, param));
}

int plugin_unload( void * plugin )
{
	if (! plugin) 	return(0);
	return( ext_plugin_unload(plugin) );
}

int target_output_as(char *output, char *filename, int options)
{
	/* called with liboutput.$OUTPUT.so as 'output',
	 * e.g. "liboutput.asm.so would be "asm" */
	char fullname[256];
	void *plugin;
	FILE *f;


	DEBUG_PrintMsg("OUTPUT TARGET as %s\n", output);
	sprintf(fullname, "output.%s", output);

	/* open output file for writing */
	if ( filename ) {
		f = fopen(filename, "w");
		if ( ! f )
			return (sys_set_lasterr(9030));

	} else {
		f = stdout;
	}
	plugin = plugin_load(fullname, options);
	if ( plugin ) {
		plugin_exec_main( plugin, f );
		plugin_unload( plugin );
	}
	/* else error */

	if ( filename ) {
		fclose( f );
	}
	return (1);
}
int target_set_size(int size);
int target_set_entry(long rva);
const char * target_arch() { return(target.arch.name); }
const char * target_asm(){ return(target.assembler.name); }
const char * target_format(){ return(target.format.name); }
const char * target_lang(){ return(target.lang.name); }
const char * target_comp(){ return(target.comp.name); }
const char * target_os(){ return(target.os.name); }
int target_os_type(){return (ext_os->type);}
