#include <api/api_main.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;

/* API sections:
 *    11. Functions            :: Creating/manipulating FUNCTION objects
 *    12. Inline               :: Creating/manipulating INLINE functions
 */


/* ---------------------------------------------------------------Functions */
int func_new(long rva, char *name, int size, int flags)
{
	struct address a = {0};
	struct function f = {0}, f2 = {0};
	struct name n = {0};
	char buf[16];

	/* Do not create a function if one already exists! */
	/* Make sure fn doesn't exist, and rva is not in another fn */

	if ( bdb_find_closest_prev(FUNCTION_RVA, &rva, &f) &&
	     bdb_index_next(FUNCTION_RVA, &f2) ) {
		if (  f.rva == rva  || 			/* new func exists */
			f.rva + f.size >= rva ||	/* new func in other func */
			(f2.rva > rva && f2.rva - 8 < rva)	 ) {
			return(0);
		}
	}

	memset( &f, 0, sizeof(struct function) );
	f.rva = rva;
	f.size = (size > 0) ? size : 1;
	/* Create address ... this will change flags on an existng addr */
	if (bdb_index_find( ADDRESS_RVA, &rva, &a)) {
		a.flags |= ADDR_CODE | ADDR_FUNCTION | ADDR_NAME;
		if ( flags & NAME_SYMBOL )	
			a.flags |= ADDR_SYMBOL;
		bdb_record_update(ADDRESS_RVA, &rva, &a);
	} else {
		a.flags = ADDR_CODE | ADDR_FUNCTION | ADDR_NAME;
		if ( flags & NAME_SYMBOL )	
			a.flags |= ADDR_SYMBOL;
		/* 1 for size and -1 for pa are "defaults" */
		addr_new( rva, 1, ADDR_UNK_PA, a.flags );
	}

	if (! bdb_record_insert(FUNCTION, &f) ) {
		return (sys_set_lasterr(db_error()));
	}
	/* create name for addr */
	if (bdb_index_find(NAME_RVA, &rva, &n)) {
		/* Preserve existing symbol and user names */
		if ((n.type & NAME_USER) || (n.type & NAME_SYMBOL)) {
			n.type |= NAME_SUB;
			bdb_record_update(NAME_RVA, &rva, &n);
			return (1);	/* Bail out early, we are done naming */
		}
	}

	if (!flags)
		flags = NAME_AUTO | NAME_SUB;
	else
		flags |= NAME_SUB;

	if (!name) {
		/* yeah, we provide this in name_new_default too, big deal */
		sprintf(buf, "sub_%08X", rva);
		name = buf;
	}

	name_new(rva, name, flags);

	return (1);
}

int func_get_name(long rva, char *buf)
{
	struct name n;

	if (d_keyfind(NAME_RVA, &rva) == S_OKAY) {
		d_recread(&n);
		strncpy(buf, n.text, 63);
		return (1);
	}
	return (sys_set_lasterr(4310));
}

unsigned long func_get_start(long rva)
{
	struct function f;

	if (d_keyfind(FUNCTION_RVA, &rva) == S_NOTFOUND) {
		d_keynext(FUNCTION_RVA);
		d_keyprev(FUNCTION_RVA);
	}
	if (db_status != S_OKAY)
		return (sys_set_lasterr(db_error()));
	d_recread(&f);
	return (f.rva);
}

unsigned long func_get_end(long rva)
{
	struct function f;

	if (d_keyfind(FUNCTION_RVA, &rva) == S_NOTFOUND) {
		d_keynext(FUNCTION_RVA);
		d_keyprev(FUNCTION_RVA);
	}
	if (db_status != S_OKAY)
		return (sys_set_lasterr(db_error()));
	d_recread(&f);
	return (f.rva + f.size);
}

unsigned long func_get_by_name(char *name)
{
	int rva;

	rva = addr_get_by_name(name);
	if (rva && d_keyfind(FUNCTION_RVA, &rva) == S_OKAY) {
		return (rva);
	}

	return (sys_set_lasterr(4320));
}

int func_set_name(long rva, char *name)
{
	struct function f;

	if (d_keyfind(FUNCTION_RVA, &rva) != S_OKAY)
		return (sys_set_lasterr(4310));
	return (name_new(rva, name, NAME_AUTO));
}

int func_set_size(long rva, int size)
{
	struct function f;

	if (d_keyfind(FUNCTION_RVA, &rva) == S_OKAY) {
		d_recread(&f);
		f.size = size;
		d_recwrite(&f);
		return (1);
	}
	return (sys_set_lasterr(4310));
}

int func_foreach( BAPI_CALLBACK fn, void *arg){
	struct function f;
	int cont;
	void *state;

	cont = bdb_table_first(FUNCTION, &f);
	while( cont ) {
		state = db_save_state();
		(*fn)(&f, arg);
		db_restore_state( state );
		cont = bdb_table_next( FUNCTION, &f );
	}
	return(1);
}

int func_foreach_addr(struct function *f, BAPI_CALLBACK fn, void *arg)
{
	struct address a;
	int cont, *state;

	cont = bdb_find_closest_next(ADDRESS_RVA, &f->rva, &a);
	while (cont && a.rva >= f->rva && a.rva < (f->rva + f->size)) {
		state = db_save_state();
		(*fn) (&a, arg);
		db_restore_state( state );
		cont = bdb_index_next(ADDRESS_RVA, &a);
	}
	return (1);
}

int func_foreach_code(struct function *f, BAPI_CALLBACK fn, void *arg)
{
	struct code c;
	int cont, *state;

	cont = bdb_find_closest_next(CODE_RVA, &f->rva, &c);
	while (cont && c.rva >= f->rva && c.rva < (f->rva + f->size)) {
		state = db_save_state();
		(*fn) (&c, arg);
		db_restore_state( state );
		cont = bdb_index_next(CODE_RVA, &c);
	}
	return (1);
}

int func_gen_args( struct function *f ) {
	if (! f) return(0);
	return( ext_func_genargs(f) );
}

int func_gen_effect( struct function *f ) {
	if (! f) return(0);
	return( ext_func_geneffect(f) );
}
int func_gen_locals( struct function *f ){
	return(0);
}
/* return prototype for the given function */
int func_gen_proto( struct function *f, char *buf, int len ){
	return( ext_sprint_hll_proto(buf, len, f->id) );
}



/* ---------------------------------------------------------------Inline */
/* Flags to add : paginate, hide libs, hide inline */
/* Note : flags should be changed in BC to avoid parsing :) */
