#include <api/api_main.h>
#include <extension.h>
#include <debug.h>
#include <vm.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;
unsigned char *disasm_buf = NULL;
int disasm_follow_xrefs = 0;

/* API sections:
 *     4. Disassembly         :: All disassembly passes
 */


struct code *curr_inst;
/* ---------------------------------------------------------------Disassembly */
int target_set_header(char *strHeader)
{
/* This just stores a character representation of the file header 
 * in memory -- formatting is left up to the file format parser
 * script.
 */
	if (target.info.header)
		free(target.info.header);
	target.info.header = strdup(strHeader);
	if (!target.info.header)
		return (sys_set_lasterr(9050));
	disasm_env.flags |= DB_MOD;
	return (1);
}

char *target_header()
{
	return (target.info.header);
}

static int deal_with_addr(unsigned long rva, struct code *code, int xrefType,
		   int nametype)
{
	int addrType;

	if (rva && addr_is_valid(rva)) {
		addrType = ADDR_DATA;

		if (xrefType & XREF_EXEC) {
			/* This is an executable addr */
			addrType = ADDR_CODE;
		}

		/* create an address object for it if none exists */
		if (d_keyfind(ADDRESS_RVA, &rva) != S_OKAY)
			addr_new(rva, 1, 0, addrType);

		/* If not already named, name it */
		if (d_keyfind(NAME_RVA, &rva) != S_OKAY) {
			name_new_default( rva, nametype );
		}

		if ( addrType == ADDR_CODE && ( 
		    (code->mnemType & INS_TYPE_MASK) == INS_CALL ||
		    (code->mnemType & INS_TYPE_MASK) == INS_CALLCC ) ) {
			/* make target a function */
			func_new(rva, NULL, 0, 0);
		}

		/* Make an xref from this instruction to the rva */
		if (! xref_new(code->rva, rva, xrefType) ) {
			return(0);
		}
	}
	return (1);
}

static int analyze_curr_op(struct code *code, long *op, int *type)
{
/* This finds absolute and rfelative address references in an opcode,
 * creates an address object for that reference, creates an xref to
 * that address object, and performs location naming/subroutine
 * definition for references that are jmp or call targets */
	struct address *a;
	struct comment *cmt;
	int xrefType, change = 0;
	char buf[64] = "\0";
	long rva = 0;

	/* Mask the XREF Type [rwx] from the instruction */
	xrefType = *type & OP_PERM_MASK;

	/* first do a check and be sure this is not a valid offset disguised as
	 * an IMM/REL [e.g. 'pop [addr]' is considered an IMM] */
	if (!(*type & OP_PTR) && addr_is_valid_code(*op)) {
		*type &= ~OP_TYPE_MASK;	/* 0xFF0FF0FF;  clear Type */
		*type |= OP_ADDR;
	}

	switch (*type & OP_TYPE_MASK) {
	case OP_REG:
		break;
	case OP_EXPR:
		/* for now, ignore registers && indirect addresses */
		break;
	case OP_IMM:
		/* If IMM is only read, it is a constant */
		if (!(xrefType & XREF_EXEC) && !(xrefType & XREF_WRITE))
			break;
		/* else fall through to 'rel offset' */
	case OP_REL:
/* TODO: For DATA, associate default data type with target rva */
		/* RELATIVE OFFSET Convert offset to RVA */
		/* NOTE: skip rel offsets of 0 */
		if (*op && (a = GetAddressObject(code->rva))) {
			/* add IMM value to end of current insn [rva + size] */
			rva = *op + code->rva + a->size;

			/* note the change in instruction comment */
			sprintf(buf, "(0x%X was %+d)", rva, *op);
			addr_set_comment(code->rva, comment_new(buf, CMT_AUTO));

			/* change the operand to the new rva */
			*op = rva;
			*type &= ~OP_TYPE_MASK;	/* 0xFF0FF0FF;  clear Type */
			*type |= OP_ADDR;	/* Set type to Absolute Addr */
			free(a);

		} else if (*op) {
			/* This shouldn't happen */
			DEBUG_PrintMsg("Horrible error in analyze_curr_op! Addr"
				       " %x for CODE struct doesn't exist\n",
				       code->rva);
			break;
		}
		/* fall through to handle new address */
	case OP_ADDR:
/* TODO: For DATA, associate default data type with rva */
		rva = *op;
		deal_with_addr(rva, code, xrefType, NAME_NEWLOC);
		break;
	case OP_PTR:
/* TODO: For DATA, associate default pointer type with pointer rva */
/* TODO: For DATA, associate default data type with deferenced rva */
		rva = *op;
		/* first, create address object for pointer itself */
		deal_with_addr(rva, code, xrefType, NAME_NEWPTR);

		/* next, create address for rva contained in the pointer */

		/* find section rva is in, get offset from start of section,
		 * read bytes from file, use that as rva.Put op in brackets. */
		if ((rva = addr_pa(rva)) && rva < target.info.size) {
			/* get addr in pointer from target image, if possible */
			encpy((char *) &rva, target.image + rva,
			      ext_arch->sz_addr);
			if (addr_is_valid(rva))
				xref_new(code->rva, rva, xrefType);
		}
		break;
	default:
		break;
	}

	if ((*type & OP_MOD_MASK & OP_STRING) &&
	    (*type & OP_TYPE_MASK) == OP_ADDR    ) {
/* TODO: string recognition? e.g. if OP_STRING_PTR on rep scasb? */
		/* this will do nothing if OP is not a valid rva */
		rva = *op;
		str_new(rva);
	}

	return (1);
}

static int handle_code_vm( struct code *code ) {
	/* here, we want to handle the stack for the VM, and
	 * also to disassemble_forward for all dynamic calls. 
	 * OK, this really should only be called from disasm_forward,
	 * but it is useful so we will leave it... */
	struct VM_DATA data = {0};

	switch ( code->mnemType & INS_TYPE_MASK ) {
		case INS_CALL:
		case INS_CALLCC:
			vm_frame_stack();
		case INS_BRANCH:
		case INS_BRANCHCC:
		case INS_LOOP:
			if ( (code->destType & OP_TYPE_MASK) == OP_REG ) {
				vm_read_reg( code->dest, &data );
				if ( ! data.flags ) {
					/* is this appropriate? Useful tho */
					xref_new(code->rva, data.u.l, 
							XREF_EXEC);
					/* disasm forward from val in reg */
					if ( disasm_follow_xrefs ){
						disasm_forward( data.u.l );
					}
				//} else if ( data.flags & VM_STACK_PTR ) {
				}
			}
			break;
		case INS_RET:
			vm_unframe_stack();
			break;
		case INS_PUSH:
			vm_push_operand( code->dest, code->destType );
			break;
		case INS_POP:
			vm_pop_operand( code->dest, code->destType );
			break;
		case INS_SUB:
			/* if ( code->destType == OP_REG && 
			 * code->dest == ext_arch->SP )...*/
		case INS_ADD:
			/* work... save it for later ;) */
			/* this should really be part of vm_reg_op anyways */
			break;
		case INS_MOV:
			/* all we really care about is MOVs into registers */
			if ( (code->destType & OP_TYPE_MASK) == OP_REG ) {
				if ((code->srcType & OP_TYPE_MASK) == OP_REG) {
					vm_read_reg( code->dest, &data );
					vm_write_reg( code->dest, &data );
				} else if ( (code->srcType & OP_TYPE_MASK) == 
						OP_IMM ||
					    (code->srcType & OP_TYPE_MASK) == 
						OP_ADDR ) {
					data.u.l = code->src;
					vm_write_reg( code->dest, &data );
				}
			}
		default:
			break;
	}
	return(1);
}

static int analyze_curr_inst(struct code *code)
{
	struct code_effect e;
	/* add support for arch-specific comments, e.g. */
	/* If disasm->comment_get(mnemonic) ... */

	/* Get standard reg effects */
	code_effect_gen( code );

	/* Get Operand Types */
	if (code->destType)
		analyze_curr_op(code, &code->dest, &(code->destType));
	if (code->srcType)
		analyze_curr_op(code, &code->src, &(code->srcType));
	if (code->auxType)
		analyze_curr_op(code, &code->aux, &(code->auxType));

/* Get Instruction Type */

/* Make Code object */
	if (d_fillnew(CODE, code) != S_OKAY) {
		return (sys_set_lasterr(db_error()));
	}

	/* do VM stuff */
	handle_code_vm( code );

	/* is this a system call? */
	if ( (code->mnemType & INS_GROUP_MASK) == INS_TRAPS ) { 
		if ( sysref_fromcode( code ) ) {
			/* get syscall param_type */
			/* get num (stack) params to syscall */
			/* do (that_many|genregs)  reg_states */
			;
		}
	} 

	return (1);
}

int disasm_target(char *disasm, void *param)
{
	/* called with libdisasm.$DISASM.so as 'disasm',
	 * e.g. "libdisasm.full.so would be "full" */
	char fullname[256] = { 0 };
	void *plugin;


	DEBUG_PrintMsg("Disassemble %s on TARGET ", disasm);
	DEBUG_PrintMsg("entry %08X\n", target.info.entry);
	sprintf(fullname, "disasm.%s", disasm);
	plugin = plugin_load(fullname, 0);
	if (plugin) {
		ext_plugin_main(plugin, param);
		plugin_unload(plugin);
	}
/* else error */
	return (1);
}

int disasm_address(unsigned long pa, struct code *c, unsigned long rva)
{
/* pa  = offset from image start  
 * c   = code struct filled with info
 * rva = RVA to disassemble */
	struct address a = {0};
	struct code code = {0};
	int bytes;



	if ( bdb_find_closest_prev( CODE_RVA, &rva, &code ) &&
		bdb_index_find( ADDRESS_RVA, &code.rva, &a ) ) {
		if ( code.rva == rva ) {
			if ( disasm_prefs.options & DISASM_REDO ) {
				code_del( code.rva );
			} else {
				return( sys_set_lasterr(4120) );
			}
		} else if ( a.rva <= rva && (a.rva + a.size) > rva ) {
			if ( disasm_prefs.options & DISASM_REDO ) {
				code_del( code.rva );
			} else {
				return( sys_set_lasterr(4130) );
			}
		}
	}
	

	/* IF this is an invalid Physical Address, BailOut */
	if (pa < 0 || pa > (long) target.info.size)
		return (-1);
	/* ELSE call disassembler (ARCH)  extension */
	/* prepare code buffer */
	if (! disasm_buf) disasm_buf = malloc( ext_arch->sz_inst );
	memset(disasm_buf, 0, ext_arch->sz_inst );
	/* use buffer to protect from overrunning mmap of file */
	bytes = target.info.size - pa; 	/* reuse bytes ;) */	
	bytes = ( bytes > 20 ) ? 20 : bytes;	
	memcpy( disasm_buf, target.image + pa, bytes );  
	bytes = ext_disasm_addr(disasm_buf, 0, c, rva);
	//bytes = ext_disasm_addr(target.image + pa, 0, c, rva);

	/* IF this is an invalid instruction, BailOut */
	if (bytes < 1)
		return (-1);

	/* ELSE Make an address object for rva */
	c->rva = rva;
	addr_new(c->rva, bytes, pa, ADDR_CODE);	/* Define handles dupes */

	/* Analyze current instr and create a CODE object */
	analyze_curr_inst(c);

	return (bytes);		/* return size of instruction */
}

int disasm_section(char *name)
{
	struct section s;
	struct address a;
	char msg[128];

	if ((ext_arch->ext.flags & EXT_STATUS_MASK) != EXT_INIT)
		target_set_arch(target.arch.name, 0);
	if (bdb_index_find(SECTION_NAME, &name, &s)) {
		sprintf(msg, "Disassembling section %s: %X\n", name, s.rva);
		sys_msg(msg);
		disasm_range(s.rva, s.size);
		sys_msg("Done.\n");
	} else
		return (0);	/* error will be set by GetSectionObject */
	disasm_env.flags |= DB_MOD;
	return (1);
}

int disasm_range(unsigned long rva, int size)
{
/* rva must have an address record */
	struct address a;
	struct code code;
	int bytes, len;

	disasm_follow_xrefs = 0;
	if ((ext_arch->ext.flags & EXT_STATUS_MASK) != EXT_INIT)
		target_set_arch(target.arch.name, 0);
	disasm_env.flags |= DB_MOD;
	if (! bdb_index_find(ADDRESS_RVA, &rva, &a) ){
		/* try for closest address */
		if (! bdb_find_closest_next(ADDRESS_RVA, &rva, &a) ) {
			return (0);	/* last_err already set */
		}
		size -= (a.rva - rva);
	}
	for (bytes = 0; bytes <= size;) {
		memset(&code, 0, sizeof (struct code));
		code.rva = rva + bytes;
		len = disasm_address(a.pa + bytes, &code, code.rva);
		if (len < 1) {
			bytes++;
			break;
		} else
			bytes += len;
	}
	return (1);
}

static unsigned long disasm_branch_dest( struct code *c, int size ) {
	int type;
	unsigned long rva = 0;

	if ( ! c ) 	return(0);
	type = c->destType & OP_TYPE_MASK;
	switch ( type ) {
		case OP_REG:
			/* skip registers for now */
			break;
		case OP_PTR:
			rva = c->dest;
			/* 1. get pa for rva */
			if ((rva = addr_pa(rva)) && rva < target.info.size) {
				/* 2. get dword [pointer] stored at pa */
				encpy((char *) &rva, target.image + rva, 
						ext_arch->sz_addr);
			}
			break;
		case OP_IMM:
		case OP_REL:
			rva = c->rva + size + c->dest;
			break;
		default:
			rva = c->dest;
	}
	return( rva );
}

int disasm_addr_switch(struct code *c, int size)
{
	int cont = 1;
	char tmp_op[32];	/* temporary operand */
	long new_rva = 0;

	/* At this point, addr_new and AnalyzeInst have been done */
	switch (c->mnemType & INS_TYPE_MASK) {
		case INS_BRANCH:
			cont = 0;
		case INS_BRANCHCC:
		case INS_CALL:
		case INS_CALLCC:
			new_rva = disasm_branch_dest( c, size );
			break;
		case INS_RET:
		case INS_TRET:
			cont = 0;
		default:
			break;
	}

	/* Follow branch if one was detected */
	sys_visual("\b-");		/* step 3 on this stack level */
	/* put a stack bounds check in here ??? */
	/* TODO : addr_is_valid and such? */
	if (new_rva && new_rva != c->rva) {
		disasm_forward(new_rva);
	}

	return (cont);
}

int disasm_forward_addr(struct address *a, struct section *s)
{
	struct code code;
	int bytes = 0, size = 0, cont = 1;

	sys_visual(" \b+");	/* Display a visual stack */
	while (cont) {
		memset(&code, 0, sizeof (struct code));
		code.rva = a->rva + bytes;
		sys_visual("\b|");	/* step 1 of this stack level */
		size = disasm_address(a->pa + bytes, &code, code.rva);
		sys_visual("\b/");	/* step 2 on this stack level */
		if (size < 1) {
			cont = 0;
			continue;
		} else {
			bytes += size;
			cont = disasm_addr_switch(&code, size);
		}

		sys_visual("\b/");	/* step 4 on this stack level */
		if (a->rva + bytes >= s->rva + s->size)
			cont = 0;
	}			// end while (cont)
	sys_visual("\b \b");	/* undo `sys_msg("|")` */
	return (1);
}

int disasm_forward(unsigned long rva)
{
	struct section sec = {0}, snext = {0};
	struct address addr;
	char buf[32];
	void *state;

	disasm_follow_xrefs = 1;
	if ((ext_arch->ext.flags & EXT_STATUS_MASK) != EXT_INIT)
		target_set_arch(target.arch.name, 0);
	state = db_save_state();
	if (sec_get(rva, &sec)) {
		/* Get starting addr of segment to use as a base for pa */
		bdb_find_closest_prev(ADDRESS_RVA, &sec.rva, &addr);

		addr.pa += (rva - addr.rva);
		addr.rva = rva;
		addr.flags = ADDR_CODE;
		disasm_forward_addr(&addr, &sec);
	} else {
		/* this happens with GNU linkers. Basically, we find the start
		 * of the next section and create an address linking the end of
		 * this section with the start of the next one */
		bdb_find_closest_prev(SECTION_RVA, &rva, &sec);
		bdb_find_closest_next(SECTION_RVA, &rva, &snext); 
		if ( snext.rva <= sec.rva ) {
			/* this is not a valid address in the program */
			db_restore_state(state);
			return(sys_set_lasterr(4250));
		}
		bdb_find_closest_prev(ADDRESS_RVA, &rva, &addr);
		/* create new address object on the fly */
		addr.rva += addr.size; 
		addr.pa += addr.size;
		addr.size = snext.rva - addr.rva;
		memset( &addr.dataType, 0, 5 * sizeof(long));
		bdb_record_insert( ADDRESS, &addr );
		/* resize section */
		sec.size += addr.size;
		bdb_record_update(SECTION_RVA, &sec.rva, &sec);
		disasm_forward_addr(&addr, &sec);
	}
	disasm_env.flags |= DB_MOD;
	db_restore_state(state);
	return (1);
}


/* Convenience functions -- low-level wrappers to the disasm_lib fn's */
unsigned long disasm_find_pat_in_range(unsigned long rva, int size, int type)
{
	struct address a;
	int cont;

	cont = bdb_find_closest_prev(ADDRESS_RVA, &rva, &a);
	while (cont && a.rva - rva < size) {
		if (ext_code_pat(a.rva, type))
			return (a.rva);
		cont = bdb_index_next(ADDRESS_RVA, &a);
	}
	return (0);
}

unsigned long disasm_find_func_in_range(unsigned long rva, int size)
{
	unsigned long start;
	struct address a;
	struct code c;

	while ((start = disasm_find_pat_in_range(rva, size, FUNCTION_PROLOGUE))){
		bdb_index_find( ADDRESS_RVA, &start, &a );
		bdb_index_prev( ADDRESS_RVA, &a );
		if ( a.rva + a.size >= start ) {
			/* there is an address directly before this one */
			if ( bdb_index_find( CODE_RVA, &a.rva, &c) ) {
				if ( (c.mnemType & INS_TYPE_MASK) == INS_RET    ||
				 	 c.mnemType & INS_TYPE_MASK == INS_BRANCH ||
				  	 c.mnemType & INS_TYPE_MASK == INS_TRET   ||
					 c.mnemType & INS_TYPE_MASK == INS_NOP     ) {
					/* previous CODE is a JMP, RET or IRET */
					return( start );
				}
				/* else we are in the middle of code or data! */
			} else 		/* no preceding code */
				return(start);
		} else			/* no preceding address */
			return(start);

		/* dammit, try again starting from here */
		bdb_index_next( ADDRESS_RVA, &a );
		start = a.rva + a.size;
		size -= ( start - rva );
		rva = start;
	}
}

static unsigned long disasm_find_epilogue(unsigned long rva, unsigned long id){
	struct address a;
	struct code c;
	int cont, type;
	unsigned long branch_end = 0, next_end = 0;
	
	cont = bdb_index_find(ADDRESS_RVA, &rva, &a);
	while (cont) {
		/* while we're here, aissgn fn to the code */
		code_set_func(a.rva, id);
		if (ext_code_pat(a.rva, FUNCTION_EPILOGUE))
			return (a.rva);
		/* do we need to branch or anything? */
		bdb_index_find( CODE_RVA, &a.rva, &c );
		type = c.mnemType & INS_TYPE_MASK;
		if ( type == INS_BRANCH ) {
			rva = disasm_branch_dest( &c , a.size);
			if ( rva ) {
				cont = bdb_index_find( ADDRESS_RVA, &rva, &a );
			} else {
				/* cannot follow jump */
				return( rva );
			}
		} else if ( type == INS_BRANCHCC ) {
			if ( bdb_index_next(ADDRESS_RVA, &a) ) {
				next_end = disasm_find_epilogue( a.rva, id );
			}
			rva = disasm_branch_dest( &c, a.size );
			if ( rva && rva > next_end ) {
				branch_end = disasm_find_epilogue( rva, id );
			}
			return( (branch_end > next_end)? branch_end : next_end );
		} else if ( type == INS_RET ) {
			return( a.rva );
		} else {
			cont = bdb_index_next(ADDRESS_RVA, &a);
		}
	}
	return(0);
}

unsigned long disasm_find_func_end(unsigned long rva)
{
	struct function f;

	if( bdb_find_closest_prev(FUNCTION_RVA, &rva, &f) ) 
		return( disasm_find_epilogue( f.rva, f.id ) );
	return(0);
}

int disasm_byte_order()
{
	return (ext_arch->endian);
}
int disasm_addr_size()
{
	return (ext_arch->sz_addr);
}
int disasm_byte_size()
{
	return (ext_arch->sz_byte);
}
int disasm_word_size()
{
	return (ext_arch->sz_word);
}
int disasm_dword_size()
{
	return (ext_arch->sz_dword);
}
int disasm_get_sp()
{
	return (ext_arch->SP);
}
int disasm_get_ip()
{
	return (ext_arch->IP);
}
