#include <api/api_main.h>
#include <extension.h>
#include <vm.h>

extern struct DISASM_TGT target;
extern struct DISASM_ENV disasm_env;
extern struct DISASM_PREFS disasm_prefs;

/* API sections:
 *     5. Addresses            :: Creating/manipulating ADDRESS objects
 * /


/* ---------------------------------------------------------------Addresses */
#define IS_REASONABLE(x) x < 0x00100000
unsigned long addr_guess_pa( unsigned long rva ) {
	struct section s;
	struct address a;
	void *state;
	int diff;
	long rv = 0;
	
	state = db_save_state();
	
	if ( sec_get( rva, &s ) ) {
		/* base pa on the start of the last section */ 
		if (bdb_index_find(ADDRESS_RVA, &s.rva, &a)) {
				rv = a.pa + (rva - a.rva);
		}
	}
	/* base pa on the start of the previous address */
	if (! rv && bdb_find_closest_prev(ADDRESS_RVA, &rva, &a) ) {
		diff = rva - a.rva;
		if ( IS_REASONABLE(diff) ) {
			rv = a.pa + diff;
		} 	
	}
	/* give up */
	db_restore_state(state);
	return(rv);
}
#undef IS_REASONABLE
int addr_is_valid(long rva)
{
	struct section s;
	int cont;

	/* foreach section: if code, the see if addr is inside it */
	cont = bdb_table_first(SECTION, &s);
	while (cont) {
		if (rva >= s.rva && rva < s.rva + s.size)
			return (1);
		cont = bdb_table_next(SECTION, &s);
	}
	return (0);
}

int addr_is_valid_code(long rva)
{
	struct section s;
	int cont;

	/* foreach section: if code, the see if addr is inside it */
	cont = bdb_table_first(SECTION, &s);
	while (cont) {
		if (s.flags == SECTION_CODE) {
			if (rva >= s.rva && rva < s.rva + s.size)
				return (1);
		}
		cont = bdb_table_next(SECTION, &s);
	}
	return (0);
}

long addr_find_closest(long rva)
{
/* Finds the closest existing address object to 'rva'
 * Note that this only searches backwards from rva; the
 * intent is to find the address object containing rva.
 *
 * A side effect of 'keyfind' is that a subsequent call to
 * 'keynext' returns the next sorted record in the index -- 
 * so doing a 'keynext' followed by a 'keyprev' sets the current
 * record to the ADDRESS object before the rva being searhced for.
 */
	struct address address;

	if (d_keyfind(ADDRESS_RVA, &rva) != S_NOTFOUND)
		return (0);
	if (d_keynext(ADDRESS_RVA) != S_OKAY)
		return (0);
	if (d_keyprev(ADDRESS_RVA) != S_OKAY)
		return (0);
	d_recread(&address);
	if (address.rva + address.size >= rva)
		/* ADDRESS object exists containing rva */
		return (address.rva);
/* else no ADDRESS object containing rva; go ahead and make one */
	return (0);
}

int addr_merge(struct address *a, struct address *a2)
{
/* 'a' is existing address [start of merge], 'a2' addr to be created */
	struct address addr = {0};
	int cont;
	unsigned long end_rva;
	
	/* make 'a' current record */
	if ( d_keyfind(ADDRESS_RVA, &a->rva) != S_OKAY )
		return(0); 	/* horrible error */
	
	if ( a->rva < a2->rva ) {
		/* resize a, create a2 */
		a->size -= a2->rva - a->rva;

		d_recwrite(a);
		d_fillnew( ADDRESS, a2 );
	} else if ( a->rva > a2->rva ) {
		/* create a2, remove a in merge */
		d_fillnew(ADDRESS, a2);
	} else { /* a->rva == a2->rva */
		/* replace a with a2, remove rest in merge */
		d_recwrite(a2);
	}
		
	/* make sure none of the previous failed */
	if (db_status != S_OKAY) {
		return (sys_set_lasterr(db_error()));
	}
	
	/* now a2 contains the addr we start "merging" from */
	end_rva = a2->rva + a2->size;
	
	cont = d_keyfind(ADDRESS_RVA, &a2->rva);
	while (cont == S_OKAY) {
		/* remove all addresses that will be inside of new, bigger address */
		cont = d_keynext(ADDRESS_RVA);
		if ( cont == S_OKAY) {
			d_recread(&addr);
			if (addr.rva >= end_rva) {
				/* cool, this starts after 'a2' ends */
				cont = S_NOTFOUND;
			} else if ( addr.rva + addr.size < end_rva ) {
				/* ok, 'a2' absorbs this address, delete it */
				d_delete();
				code_del(addr.rva);
			} else {
				/* dammit, next addr contains part of 'a2': resize it */
				addr.size -= end_rva - addr.rva;
				addr.pa += end_rva - addr.rva;
				addr.rva = end_rva;
				//addr.flags = addr.flags & ~ADDR_SECTION;
				d_recwrite(&addr);
				code_del(addr.rva);
				cont = S_NOTFOUND;
			}
		}
	}	/* end while */

	return (1);
}


int addr_split(struct address *a, struct address *a2)
{
/* note: a is the existing addr; a2 is the to-be-created addr */
	struct address address= {0};

	/* modify existing addr so it is now address *after* rva */
	if ( a2->rva < a->rva || a2->size >= a->size ) return(0);
	
	/* make "address" the address after a2->rva */
	memcpy( &address, a, sizeof(struct address) );
	//address.flags = a->flags & ~ADDR_SECTION;
	address.rva = a2->rva + a2->size;
	address.pa = a2->pa + a2->size;
	address.size -= address.rva - a->rva;
	d_fillnew(ADDRESS, &address);
	
	d_keyfind(ADDRESS_RVA, &a->rva);
	if ( a2->rva == a->rva ) {
		/* we just use a2 for a */
		d_recwrite(a2);
	} else {
		/* make a1 smaller */
		a->size = a2->rva - a->rva;
		d_recwrite(a);

		/* create address object for rva */
		d_fillnew(ADDRESS, a2);
		if (db_status != S_OKAY) {
			return (sys_set_lasterr(db_error()));
		}
	}
	return (1);
}

int addr_new(long rva, int size, long pa, int flags)
{
/* If ADDRESS object 'rva' exists, update its flags and
 * size, resize if necessary. Otherwise, create ADDRESS object
 * for rva, resizing other ADDRESS objects as needed.
 */
	struct address addr_old= {0}, address = {0};
	void *state = NULL;
	int done = 0;
	

	/* ENTER -- save DB state and such */
	if (!size) {
#if 0
		/* size = NextAddr - curr Addr  ... is this useful? */
		if (bdb_find_closest_next(ADDRESS_RVA, &rva, &address)) {
			if (address.rva != rva) {
				size = address.rva - rva;
			} else
				size = address.size;	/* shouldn't happen */
		} else
#endif
		size = 1;	/* must be the last addr in prog? */
	}
	state = db_save_state();
	disasm_env.flags |= DB_MOD;

	/* fix variables that are blank */
	if (!flags)
		flags = ADDR_DATA;
	
	if (pa == ADDR_NO_PA) {
		pa = ADDR_INVALID;
	} else if (pa == ADDR_UNK_PA || pa > (long)target.info.size) {
		pa = (long) addr_guess_pa( rva );
	}

	/* fill address structure */
	address.rva = rva;
	address.pa = pa;
	address.flags = flags;
	address.size = size;

	if (bdb_find_closest_prev(ADDRESS_RVA, &rva, &addr_old)) {
		if ( rva == addr_old.rva ) {
			/* Possibility 1 : Address already exists */
			if (addr_old.size == size) {
				/* Possibility 1a : Address exists, is the same size */
				//d_recwrite( &address );
				done = 1;
			} else if (addr_old.size < size) {
				/* Possibility 1b : Address exists and is smaller */
				done = addr_merge(&addr_old, &address);
			} else {
				/* Possibility 1c : Address exists and is larger */
				done = addr_split(&addr_old, &address);
			}
		/* Possibility 2 : RVA contained in previous address  */
		} else if ( address.rva < (addr_old.rva + addr_old.size) ) {
			if (addr_old.rva + addr_old.size > address.rva + address.size) {
				/* 2a : Previous address contains entire ADDRESS: split it */
				done = addr_split(&addr_old, &address);
			} else {
				/* 2b : ADDRESS spans prev & next addrs; merge them */
				done = addr_merge(&addr_old, &address);
			}
		}
	}

	/* Possibility 3 : RVA extends into subsequent address  */
	if (! done ) {
		if ( bdb_find_closest_next(ADDRESS_RVA, &rva, &addr_old) ){	
			if ( addr_old.rva + addr_old.size > address.rva + address.size ) {
				/* 3a : Next address contains entire 'missing' addr: split it */
				/* resize addr_old */
				d_keyfind( ADDRESS_RVA, &addr_old.rva );
				addr_old.size -= (address.rva + address.size) - addr_old.rva;
				addr_old.pa += (address.rva + address.size) - addr_old.rva;
				addr_old.rva =  address.rva + address.size;
				d_recwrite( &addr_old );
				/* create address */
				d_fillnew(ADDRESS_RVA, &address);
			} else {
				/* 3b : ADDRESS spans multiple next addrs; merge them */
				done = addr_merge(&addr_old, &address);
			}

		/* Possibility 4: RVA touches no existing addresses */
		} else {
			/* OK then, just fill it */
			d_fillnew(ADDRESS, &address);
			if (db_status == S_OKAY)
			done = 1;
		}
	}
	/* Possibility 3 : RVA extends into subsequent address  */

	/* if still not done, something fucked up */
	if (! done ) return ( sys_set_lasterr(db_error()) );

	/* LEAVE -- Restore DB state and all that */
	if (state)
		db_restore_state(state);
	return (done);
}


int addr_del(long rva)
{
/* if keyfind(addr) remove from DB.
 * else if in other addr get rva but do not delete
 * TODO: if code, if name, if string, etc ... delete
 */
	if (!bdb_record_delete(ADDRESS_RVA, &rva))
		return (0);
	code_del(rva);
	disasm_env.flags |= DB_MOD;
	return (1);
}

int addr_exists(long rva)
{
	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY)
		return (TRUE);
	return (FALSE);
}

int addr_make_code(long rva)
{
	struct address *a;
	struct code c;
	int retval = 0;

	if (a = GetAddressObject(rva)) {
		c.rva = rva;
		/* TODO : change this to a more generic disassembly function ! */
		retval = disasm_address(a->pa, &c, c.rva);
		free(a);
	}
	return (retval);
}

int addr_make_data(long rva)
{
	struct address *a;
// if already_code 
//       remove code object, etc
//      defineaddr with new flags
	code_del(rva);
	if (a = GetAddressObject(rva)) {
		a->flags |= ADDR_DATA;
		a->flags ^= ADDR_CODE;
		d_recwrite(a);
		disasm_env.flags |= DB_MOD;
		free(a);
		return (-1);
	}
	return (0);
}

int addr_flags(long rva)
{
	struct address *a;
	int flags;

	if (a = GetAddressObject(rva)) {
		flags = a->flags;
		free(a);
		return (flags);
	}
	return (0);
}

int addr_set_flags(long rva, int flags)
{
	struct address *a;

	if (a = GetAddressObject(rva)) {
		a->flags = flags;
		d_recwrite(a);
		disasm_env.flags |= DB_MOD;
		free(a);
		return (1);
	}
	return (0);
}

/* ---------------------------------------------------------- sprintf's */
#define MAX_CMT_APPEND 64
int addr_gen_autocmt(struct address *a, char *buf, int len)	{
	struct code c;
	struct sysref sys;
	struct sysref_from sf = {0};
	void *state;

	if (! a) return(0);
	state = db_save_state();

	if ( a->flags & ADDR_CODE ) {
		sf.rva = a->rva;
		if ( bdb_index_find(CODE_RVA, &a->rva, &c) &&
			(c.mnemType & INS_GROUP_MASK) == INS_TRAPS ) {
			if ( bdb_find_closest_next( SYSREF_FROM, &sf, &sys) &&
				sys.rva == a->rva ) {
					/* we have a SYSREF */
				sysref_sprint( &sys, buf, len);
			}
		}
	}

	db_restore_state(state);
	return(1);
}

int sprint_seg(char *buf, int seg)
{
	seg = seg >> 16;
	snprintf(buf, 64, "%s%s:", ext_asm->reg_pre,
		 vm_get_reg_name(ext_arch->reg_seg + seg - 1));
	return (strlen(buf));
}


int sprint_op(char *buf, char *cmt, long op, int flg, int count)
{
	char op_sign = '+', tmp[64] = {0};
	struct name n;
	struct string *str;
	struct constant c;
	int op_type = flg & OP_TYPE_MASK;
	int op_perm = flg & OP_PERM_MASK;
	int op_flag = flg & OP_MOD_MASK;
	int seg = flg & OP_SEG_MASK;
	int rv = 0;

	/* switch OPERAND TYPE */
	switch (op_type) {

	case OP_PTR:
	case OP_ADDR:
		/* check for string pointer */
		if (str = GetStringObject(op)) {
			snprintf(cmt, MAX_CMT_APPEND, "%s StrRef(%08X):\"%s\"",
				 cmt, op, str->text);
			cmt[strlen(cmt) - 1] = '"';	/* just inn case */
			free(str);
		} else if (name_get(op, n.text)) {
			/*    check if name */
			strncat(tmp, n.text, 64);
		} else if (func_get_name(op, n.text)) {
			/* this will handle imports too ! */
			strncat(tmp, n.text, 64);
		} else
			snprintf(tmp, 64, "%08X", op);
		break;
	case OP_REG:
		if (seg)
			rv = sprint_seg(tmp, seg);
		snprintf(tmp + rv, 64 - rv, "%s%s", ext_asm->reg_pre,
			 	vm_get_reg_name(op));
		break;
	case OP_EXPR:
		if (seg)
			rv = sprint_seg(tmp, seg);
		if (!addrexp_get_str(op, tmp + rv, 64 - rv))
			strcpy(tmp, "invalid addr_expr");
		break;
	case OP_REL:
		if (op < 0) {
			op_sign = '-';
			op *= -1;
		}
		snprintf(tmp, 64, "%c0x%X", op_sign, op);
		break;
	case OP_OFF:
		/* add checking for names etc here */
		if (seg)
			rv = sprint_seg(tmp, seg);
		snprintf(tmp + rv, 64 - rv, "0x%08X", op);
		break;
	case OP_IMM:
		/*  TODO:  check if constant */
		if (  flg & OP_SIGNED ) {
			if (op < 0) {
				op *= -1;
				snprintf(tmp, 64, "%s-0x%X",ext_asm->imm_pre,op);
			} else {
				snprintf(tmp, 64, "%s0x%X", ext_asm->imm_pre,op);
			}
		} else { 
			snprintf(tmp, 64, "%s0x%X", ext_asm->imm_pre, (unsigned long)op);
		}
		break;
	default:
		snprintf(tmp, 64, "0x%X", op);
		break;
	}
	strncat(buf, tmp, (count ? count : 32));
	return (strlen(buf));
}

/* TODO Notes:
 *      - Get Comment Char from assembler 
 */
int asmsprintf(char *buf, char *format, struct address *addr)
{
/* This takes a printf-style format string and uses it to
 * format information about an address to a buffer. Format
 * specifiers are:
 *    %a - rva               %p - pa
 *    %m - mnemonic          %n - name
 *    %d - dest operand      %c - comment
 *    %s - source operand    %x - xrefs
 *    %t - third operand :)
 *    %b - raw bytes         %S - section
 *
 *    %, - cond. comma       %^ - cond. newline
 *    %; - cond. semicolon   %: - cond. colon
 * Notes:
 *      buf will be cleared in this function.
 *      'cond comma' will only be printed between two params
 *      'cond colon' and 'cond. newline' will only be printed 
 *       when text precedes them in the string.
 *      'cond. semicolon' will only be printed when text follows it.
 * These format specifiers are not meant to make sense; they are
 * meant for the convenience of disassembler scripts ;)
 *  TODO : for at&t syntax and such:
 * %l -- instr[/op] size letter %L -- inst/op size full [e.g. 'dword']
 */
	int i, j, m, count, prev_fmt = 0;
	char *d, *d2;
	unsigned char byte;
	char cmt_append[MAX_CMT_APPEND] = { 0 };
	char xref_types[5] = { 0, 'r', 'w', 0, 'x' };
	char delim[] = ", ", delim2[] = "; ", none[] = "\0";
	struct code *c;
	struct xref x = { 0 };
	struct name n = { 0 };
	struct comment cmt = { 0 };
	struct section *s;
	void *state;

	if (!buf || !format || !addr)
		return (sys_set_lasterr(9010));
	c = (struct code *) calloc(sizeof (struct code), 1);
	if (!c)
		return (sys_set_lasterr(9050));
	
	buf[0] = 0;		//just in case
	d = d2 = none;

	state = db_save_state();
	if ((addr->flags & ADDR_CODE) && ! bdb_index_find(CODE_RVA,&addr->rva,c)){
		/* fix this since we know about it :) */
		addr->flags &= ~ADDR_CODE;
		addr->flags |= ADDR_DATA;
		bdb_record_update(ADDRESS_RVA, &addr->rva, addr);
		db_restore_state( state );
		return(sys_set_lasterr(4022));
	}

	/* generate auto comments (syscalls for code n such) */
	addr_gen_autocmt(addr, cmt_append, MAX_CMT_APPEND);	

	for (i = 0; i < strlen(format); i++) {
		if (format[i] != '%') {
			sprintf(buf, "%s%c", buf, format[i]);
		} else {
			i++;
			count = 0;
			while (format[i] >= '0' && format[i] <= '9') {
				count *= 10;
				count += (format[i] - '0');
				i++;
			}
			switch (format[i]) {
			case 'a':
				strncat(buf, d, 2);
				sprintf(buf, "%s%08X", buf, addr->rva);
				prev_fmt = 1;
				d = none;
				break;
			case 'b':
				strncat(buf, d, 2);
				for (j = 0; j < (count ? count : 4); j++) {
					strncat(buf, d2, 2);
					if (j < addr->size) {
						if ((addr->pa + j) <
						    target.info.size) byte =
							    (unsigned char)
							    target.image[addr->
									 pa +
									 j];
						else
							byte = 0;	/* byte is in BSS */
						sprintf(buf, "%s%02X ", buf,
							byte);
					} else
						strcat(buf, "   ");
				}
				if (addr->size > j + 1) {
					/* if '+', add a comment stating the true size */
					strcat(buf, "+");
					snprintf(cmt_append, MAX_CMT_APPEND,
						 "%s(Addr of %d bytes)",
						 cmt_append, addr->size);
				} else
					strcat(buf, " ");
				prev_fmt = 1;
				d = none;
				break;
			case 'm':
				if (c->mnemonic[0]) {
					strncat(buf, d, 2);
					strncat(buf, c->mnemonic,
						(count ? count : 8));
					prev_fmt = 1;
				} else
					prev_fmt = 0;
				d = none;
				break;
			case 's':
				if (c->srcType) {
					strncat(buf, d, 2);
					sprint_op(buf, cmt_append, c->src,
						  c->srcType, count);
					prev_fmt = 1;
				} else
					prev_fmt = 0;
				d = none;
				break;
			case 'd':
				if (c->destType) {
					strncat(buf, d, 2);
					sprint_op(buf, cmt_append, c->dest,
						  c->destType, count);
					prev_fmt = 1;
				} else
					prev_fmt = 0;
				d = none;
				break;
			case 't':
				if (c->auxType) {
					strncat(buf, d, 2);
					sprint_op(buf, cmt_append, c->aux,
						  c->auxType, count);
					prev_fmt = 1;
				} else
					prev_fmt = 0;
				d = none;
				break;
			case 'p':
				strncat(buf, d, 2);
				sprintf(buf, "%s%08X", buf, addr->pa);
				prev_fmt = 1;
				d = none;
				break;
			case 'n':
				strncat(buf, d, 2);
				if (name_get(addr->rva, n.text)) {
					strncat(buf, d2, 2);
					strncat(buf, n.text,
						(count ? count : 32));
					prev_fmt = 1;
					d2 = none;
				} else
					prev_fmt = 0;
				d = none;
				break;
			case 'c':
				strncat(buf, d, 2);
				if (addr->comment &&
				    bdb_index_find(COMMENT_ID, &addr->comment,
						  &cmt)) {
					strncat(buf, d2, 2);
					strcat(buf, ext_asm->comment);
					strncat(buf, cmt.text,
						(count ? count : 64));
					if (cmt_append[0])
						strncat(buf, cmt_append,
							(count ? count : 64));
					//strncat(buf, cmt_append, (count ? count : 64) - 
					//                          strlen(cmt.text) ); 
					prev_fmt = 1;
					d2 = none;
				} else if (cmt_append[0]) {
					strcat(buf, ext_asm->comment);
					strncat(buf, cmt_append,
						(count ? count : 64));
					prev_fmt = 1;
					d2 = none;
				} else
					prev_fmt = 0;
				d = none;
				break;
			case 'x':
				m = count ? count : 4;
				strncat(buf, d, 2);
				if (bdb_index_find(FROM_RVA, &addr->rva, &x)) {
					strncat(buf, d2, 2);
					strcat(buf, " xrefs:");
					while (m > 0) {
						sprintf(buf, "%s >%08X[%c]",
							buf, x.to_rva,
							xref_types[x.type]);
						if (bdb_index_next(FROM_RVA, &x)
						    && x.from_rva == addr->rva)
							m--;
						else
							m = 0;
					}
				}
				if (bdb_index_find(TO_RVA, &addr->rva, &x)) {
					if (m) {
						strncat(buf, d2, 2);
						strcat(buf, " xrefs:");
					}
					m = count ? count : 4;
					while (m > 0) {
						sprintf(buf, "%s <%08X[%c]",
							buf, x.from_rva,
							xref_types[x.type]);
						if (bdb_index_next(TO_RVA, &x)
						    && x.to_rva == addr->rva)
							m--;
						else
							m = 0;
					}
				}
				if (!m) {
					d2 = none;
					prev_fmt = 1;
				} else
					prev_fmt = 0;
				d = none;
				break;
			case 'S':
				//if ( s = GetSectionObject( addr->rva ) ){
				//   strncat(buf, d2, 2); 
				//   strncat(buf, s->name, (count ? count : 16));
				//   free(s);
				//}
				break;
			case ',':
				/* insert a comma only if a field follows it */
				d = delim;
				break;
			case ';':
				/* insert a comment_char only if a field follows it */
				d2 = delim2;
				break;
			case ':':
				/* insert a colon only when text precedes it */
				if (prev_fmt) {
					strncat(buf, ":", 1);
					prev_fmt = 1;
				} else
					prev_fmt = 0;
				break;
			case '^':
				/* insert a newline only when text precedes it */
				if (prev_fmt) {
					strncat(buf, "\n", 1);
					prev_fmt = 1;
				} else
					prev_fmt = 0;
				break;
			default:
				/* not a format metachar */
				strncat(buf, &format[i], 1);
				break;
			}
		}
	}

	db_restore_state( state );
	free(c);
	return (1);
}

/* -------------------------------------------------------------------- */
/* this is intended as a generic printline function -- given an address, it
 * will print using the default code or data format, and will show function
 * names, comments, etc if the user has requested it. In light of asmsprintf,
 * this is probably pretty useless */
int addr_print(long rva)
{
	char *line;
	void *state;

	if (!(line = calloc(2048, 1)))
		return (sys_set_lasterr(9050));

	if (!ext_sprint_code(rva, line, 2048, disasm_env.output)) {
		printf("!! error in address %x: %s", rva, sys_lasterr_str());
		free(line);
		return (0);
	}
	printf("%s\n", line);
	free(line);

	return (1);
}

long addr_pa(long rva)
{
	struct address a;
	struct section s;
	void *state;
	long retval = 0;

	state = db_save_state();
	if ( bdb_index_find(ADDRESS_RVA, &rva, &a) ) {
		retval = (long)a.pa;
	} else {
		retval = (long)addr_guess_pa(rva);
	}
	db_restore_state(state);
	return (retval);
}

long addr_rva(long pa)
{
	struct address a;

	if (d_keyfind(PA, &pa) != S_OKAY)
		return (sys_set_lasterr(4030));
	d_recread(&a);
	return (a.rva);
}

int addr_type(long rva)
{
/* If addr exists, return type flag.
 * Else, get section containing address return its type 
 */
	struct address addr;
	struct section sec;
	char secname[32];

	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY) {
		d_recread(&addr);
		return (addr.flags);
	} else if (db_status == S_NOTFOUND) {
		if (sec_get_by_rva(rva, secname) == -1)
			return (-1);
		if (d_keyfind(SECTION_NAME, &secname) != S_OKAY)
			return (-1);
		d_recread(&sec);
		//may be able to remove this:
		d_keyfind(ADDRESS_RVA, &rva);
		if ((sec.flags & SECTION_CODE) || (sec.flags & SECTION_EXECUTE))
			return (ADDR_CODE);
		else
			return (ADDR_DATA);
	} else
		return (sys_set_lasterr(db_error()));
}

int addr_size(long rva)
{
	struct address a;

	if (bdb_index_find(ADDRESS_RVA, &rva, &a)) {
		return(a.size);
	}
	return (0);
}

long addr_next(long rva)
{
	struct address a;

	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY &&
	    d_keynext(ADDRESS_RVA) == S_OKAY) {
		d_recread(&a);
		return (a.rva);
	} else
		return (0);
}

long addr_prev(long rva)
{
	struct address a;

	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY &&
	    d_keyprev(ADDRESS_RVA) == S_OKAY) {
		d_recread(&a);
		return (a.rva);
	} else
		return (0);
}

int addr_bytes(long rva, char *buf)
{
	struct address a;
	if (d_keyfind(ADDRESS_RVA, &rva) != S_OKAY)
		return (sys_set_lasterr(4020));
	d_recread(&a);
	if (a.pa > target.info.size) {
		memset(buf, 0, a.size);
	} else {
		memcpy(buf, target.image + a.pa, a.size);
	}
	return (a.size);
}
int addr_add_comment(long rva, char *buf)
{
	return (addr_set_comment(rva, comment_new(buf, CMT_USER)));
}

int addr_comment(long rva)
{
	struct address a;

	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY) {
		d_recread(&a);
		return (a.comment);
	}
	return (sys_set_lasterr(4020));
}

int addr_constant(long rva)
{
	struct address *a;
	int retval = -1;

	if (a = GetAddressObject(rva)) {
		retval = a->dataConst;
		free(a);
	}
	return (sys_set_lasterr(4020));
}

int addr_struct(long rva)
{
	struct address a;

	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY) {
		d_recread(&a);
		return (a.structure);
	}
	return (sys_set_lasterr(4020));
}

int addr_set_comment(long rva, int id)
{
	struct address a;

	if (!rva || !id)
		return (-1);
	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY) {
		d_recread(&a);
		a.comment = id;
		d_recwrite(&a);
		return (1);
	}
	return (sys_set_lasterr(4020));
}

int addr_set_constant(long rva, int id)
{
	/* if ADDRESS is a data object, the constant refers to
	 * its value; if ADDRESS is a code object, the constant
	 * refers to the operand with an OP_CONST type. */
	struct address a;

	if (!rva || !id)
		return (-1);
	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY) {
		if (a.flags & ADDR_CODE)
			return (sys_set_lasterr(4023));
		d_recread(&a);
		a.dataConst = id;	/* id of the constant */
		d_recwrite(&a);
		return (1);
	}
	return (sys_set_lasterr(4020));
}

int addr_set_struct(long rva, int id)
{
	struct address a;

	if (!rva || !id)
		return (-1);
	if (d_keyfind(ADDRESS_RVA, &rva) == S_OKAY) {
		d_recread(&a);
		a.structure = id;
		d_recwrite(&a);
		return (1);
	}
	return (sys_set_lasterr(4020));
}

long addr_get_by_name(char *name)
{
	struct name n;

	if (d_keyfind(NAME_TEXT, name) == S_OKAY) {
		d_recread(&n);
		return (n.rva);
	}
	return (sys_set_lasterr(4711));
}

long addr_next_code(long rva)
{
	struct address a;

	d_keyfind(ADDRESS_RVA, &rva);
	while (db_status == S_OKAY) {
		d_keynext(ADDRESS_RVA);
		d_recread(&a);
		if (a.flags & ADDR_CODE)
			return (a.rva);
	}
	return (0);
}

long addr_next_data(long rva)
{
	struct address a;

	d_keyfind(ADDRESS_RVA, &rva);
	while (db_status == S_OKAY) {
		d_keynext(ADDRESS_RVA);
		d_recread(&a);
		if (a.flags & ADDR_DATA)
			return (a.rva);
	}
	return (0);
}
/* ---------------------- byte ordering crap */
/* byte order */
int addr_get_int(char *buf);
int addr_get_short(char *buf);
/* no byte order */
int addr_data_raw();
/* byte order */
int addr_bytes();
/* no byte order */
int addr_bytes_raw();

int addr_foreach_xref(struct address *a, BAPI_CALLBACK fn, void *arg)
{
	struct xref x;
	int cont, *state;

	cont = bdb_index_find(FROM_RVA, &a->rva, &x);
	while (cont && x.from_rva == a->rva) {
		state = db_save_state();
		(*fn) (&x, arg);
		db_restore_state(state);
		cont = bdb_index_next(FROM_RVA, &x);
	}

	cont = bdb_index_find(TO_RVA, &a->rva, &x);
	while (cont && x.to_rva == a->rva) {
		(*fn) (&x, arg);
		cont = bdb_index_next(TO_RVA, &x);
	}

	return (1);
}
